<?php

namespace App\Exceptions;

use Throwable;
use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\QueryException;
use Illuminate\Http\Exceptions\ThrottleRequestsException;
use Illuminate\Session\TokenMismatchException;
use Illuminate\Validation\ValidationException;
use Symfony\Component\HttpKernel\Exception\HttpExceptionInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\Exception\MethodNotAllowedHttpException;

class Handler extends ExceptionHandler
{
    /**
     * Inputs that are never flashed to the session on validation exceptions.
     */
    protected $dontFlash = [
        'current_password',
        'password',
        'password_confirmation',
    ];

    public function register(): void
    {
        $this->reportable(function (Throwable $e) {
            // ممكن تضيف لوجيك إرسال لجهة مراقبة هنا (Sentry/NewRelic...).
        });
    }

    /**
     * Unified JSON for Validation errors.
     */
    protected function invalidJson($request, ValidationException $exception)
    {
        return response()->json([
            'success' => false,
            'message' => $exception->getMessage() ?: 'The given data was invalid.',
            'errors'  => $exception->errors(),
        ], $exception->status);
    }

    /**
     * Unified JSON for Unauthenticated.
     */
    protected function unauthenticated($request, AuthenticationException $exception)
    {
        if ($this->wantsJson($request)) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthenticated.',
            ], 401);
        }

        return redirect()->guest(route('login'));
    }

    /**
     * Main render: JSON for AJAX/Accept: application/json, otherwise fallback to default HTML.
     */
    public function render($request, Throwable $e)
    {
        if ($this->wantsJson($request)) {
            // Validation handled by invalidJson()
            if ($e instanceof ValidationException) {
                return $this->invalidJson($request, $e);
            }

            // 419: CSRF / session expired
            if ($e instanceof TokenMismatchException) {
                return $this->jsonError('Session expired, please refresh and try again.', 419, null, $e);
            }

            // 401 / 403
            if ($e instanceof AuthenticationException) {
                return $this->jsonError('Unauthenticated.', 401, null, $e);
            }
            if ($e instanceof AuthorizationException) {
                return $this->jsonError('Forbidden.', 403, null, $e);
            }

            // 404 / 405
            if ($e instanceof ModelNotFoundException || $e instanceof NotFoundHttpException) {
                return $this->jsonError('Not found.', 404, null, $e);
            }
            if ($e instanceof MethodNotAllowedHttpException) {
                return $this->jsonError('Method not allowed.', 405, null, $e);
            }

            // 429
            if ($e instanceof ThrottleRequestsException) {
                return $this->jsonError('Too many requests. Please slow down and try again.', 429, null, $e);
            }

            // 409: Integrity / FK / Unique violations
            if ($e instanceof QueryException) {
                $sqlState = $e->errorInfo[0] ?? null; // '23000' غالبًا لـ FK/unique
                if ($sqlState === '23000') {
                    return $this->jsonError('Conflict due to related records or unique constraint.', 409, null, $e);
                }
            }

            // Any other HttpException with status
            if ($e instanceof HttpExceptionInterface) {
                $status  = $e->getStatusCode();
                $message = $e->getMessage() ?: $this->defaultMessageFor($status);
                return $this->jsonError($message, $status, null, $e);
            }

            // Fallback 500
            return $this->jsonError(
                config('app.debug') ? ($e->getMessage() ?: 'Server error.') : 'Server error, please try again later.',
                500,
                null,
                $e
            );
        }

        // Web (HTML) fallback
        return parent::render($request, $e);
    }

    /**
     * Decide if we should return JSON (AJAX/SPA/Accept: application/json).
     */
    protected function wantsJson($request): bool
    {
        return $request->expectsJson()
            || $request->wantsJson()
            || $request->isJson()
            || $request->ajax();
    }

    /**
     * Small helper to build consistent JSON error responses.
     */
    protected function jsonError(string $message, int $status, ?array $errors = null, ?Throwable $e = null)
    {
        $payload = [
            'success' => false,
            'message' => $message,
        ];

        if (!empty($errors)) {
            $payload['errors'] = $errors;
        }

        // في وضع الديبج، نضيف شوية معلومات مفيدة من غير ترايس
        if (config('app.debug') && $e) {
            $payload['debug'] = [
                'exception' => class_basename($e),
                'code'      => (int)($e->getCode() ?: 0),
            ];
        }

        return response()->json($payload, $status);
    }

    /**
     * Default message for common HTTP codes (when exception message is empty).
     */
    protected function defaultMessageFor(int $status): string
    {
        return match ($status) {
            400 => 'Bad request.',
            401 => 'Unauthenticated.',
            403 => 'Forbidden.',
            404 => 'Not found.',
            405 => 'Method not allowed.',
            409 => 'Conflict.',
            419 => 'Session expired.',
            422 => 'The given data was invalid.',
            429 => 'Too many requests.',
            500 => 'Server error, please try again later.',
            default => 'Unexpected error.',
        };
    }
}
