<?php

namespace App\Http\Controllers\Api\BRNC;

use App\Http\Controllers\Controller;
use App\Models\BtnTrack;
use App\Models\LandingFormTracking;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

/**
 * Class DataController
 *
 * Provides lightweight API endpoints to fetch recent tracked data
 * for dashboard widgets (contact_us and btn_track) with basic filters.
 */
class DataController extends Controller
{
    /**
     * Fetch BRNC data for a given account/type with optional filters.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getData(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'account_name' => 'nullable|string',             // for landing_form_tracking
            'account'      => 'nullable|string',             // for btn_track
            'type'         => 'required|in:contact_us,btn_track',
            'limit'        => 'nullable|integer|min:1|max:200',
            'from'         => 'nullable|date',
            'to'           => 'nullable|date',
            'source'       => 'nullable|string|max:255',
        ]);

        $type  = $validated['type'];
        $limit = $validated['limit'] ?? 50;
        $from  = $validated['from'] ?? null;
        $to    = $validated['to']   ?? null;
        $src   = $validated['source'] ?? null;

        if ($type === 'contact_us') {
            $q = LandingFormTracking::query()
                ->when($validated['account_name'] ?? null, fn($qq, $acc) => $qq->where('account_name', $acc))
                ->when($from, fn($qq) => $qq->whereDate('created_at', '>=', $from))
                ->when($to,   fn($qq) => $qq->whereDate('created_at', '<=', $to))
                ->when($src,  fn($qq) => $qq->where('source', $src))
                ->orderBy('created_at', 'desc')
                ->limit($limit)
                ->get([
                    'id', 'name', 'email', 'services', 'mobile',
                    'platform_info', 'account_name',
                    'platform_ip_1', 'platform_ip_2',
                    'ip_country', 'ip_city', 'ip_region',
                    'source', 'medium', 'campaign', 'term', 'content',
                    'referrer', 'page_url', 'visitor_id',
                    'created_at'
                ]);

            return response()->json(['success' => true, 'type' => $type, 'data' => $q]);
        }

        // btn_track
        $q = BtnTrack::query()
            ->when($validated['account'] ?? null, fn($qq, $acc) => $qq->where('account', $acc))
            ->when($from, fn($qq) => $qq->whereDate('created_at', '>=', $from))
            ->when($to,   fn($qq) => $qq->whereDate('created_at', '<=', $to))
            ->when($src,  fn($qq) => $qq->where('source', $src))
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get([
                'id', 'account', 'platform', 'ip_1', 'ip_2',
                'click_type', 'country', 'city', 'region',
                'source', 'medium', 'campaign', 'term', 'content',
                'referrer', 'page_url', 'visitor_id',
                'created_at'
            ]);

        return response()->json(['success' => true, 'type' => $type, 'data' => $q]);
    }
}
