<?php

namespace App\Http\Controllers\Api\BRNC;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Services\TrackingService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

/**
 * Class ReserveController
 *
 * Handles reservation submits, stores full tracking (UTM, referrer, visitor_id)
 * and maps to reservations schema (phone/service/account_name/platform_info/ips...).
 */
class ReserveController extends Controller
{
    /**
     * Store a new reservation request with tracking data.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'mobile'        => 'required|string|max:21', // will map to phone
            'service'       => 'required|string',
            'account'       => 'nullable|string|max:21',
            'account_name'  => 'nullable|string|max:21',
            'platform'      => 'nullable|string|max:21',
            'platform_info' => 'nullable|string|max:21',
            'ip'            => 'nullable|string|max:41',
            'lat'           => 'nullable|string|max:21',
            'lng'           => 'nullable|string|max:21',
            'country'       => 'nullable|string|max:21',
            'city'          => 'nullable|string|max:21',
            'region'        => 'nullable|string|max:44',

            // UTM/Attribution
            'source'     => 'nullable|string|max:255',
            'medium'     => 'nullable|string|max:255',
            'campaign'   => 'nullable|string|max:255',
            'term'       => 'nullable|string|max:255',
            'content'    => 'nullable|string|max:255',
            'referrer'   => 'nullable|string|max:512',
            'page_url'   => 'nullable|string|max:512',
            'visitor_id' => 'nullable|string|max:64',
        ]);

        $merged = TrackingService::collect($request, $validated);

        $payload = [
            'phone'          => $merged['mobile'], // map
            'service'        => $merged['service'],
            'account_name'   => $merged['account_name'] ?? ($merged['account'] ?? null),
            'platform_info'  => $merged['platform_info'] ?? ($merged['platform'] ?? null),

            'platform_ip_1'  => $merged['ip'] ?? null,
            'platform_ip_2'  => $merged['platform_ip_2'] ?? null,

            'ip_lat'         => $merged['lat'] ?? null,
            'ip_lng'         => $merged['lng'] ?? null,
            'ip_country'     => $merged['country'] ?? null,
            'ip_city'        => $merged['city'] ?? null,
            'ip_region'      => $merged['region'] ?? null,

            // attribution
            'source'         => $merged['source'] ?? null,
            'medium'         => $merged['medium'] ?? null,
            'campaign'       => $merged['campaign'] ?? null,
            'term'           => $merged['term'] ?? null,
            'content'        => $merged['content'] ?? null,
            'referrer'       => $merged['referrer'] ?? null,
            'page_url'       => $merged['page_url'] ?? null,
            'visitor_id'     => $merged['visitor_id'] ?? null,
        ];

        Reservation::create($payload);

        return response()->json([
            'success' => true,
            'message' => 'Reservation stored with tracking.',
        ]);
    }
}
