<?php

namespace App\Http\Controllers\Api\BRNC;

use App\Http\Controllers\Controller;
use App\Models\NewsletterSubscriber;
use App\Services\TrackingService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

/**
 * Class SubscribeController
 *
 * Handles newsletter subscriptions with full attribution tracking.
 * Maps to newsletter_subscribers schema (account as BIGINT, ip_1/ip_2...).
 */
class SubscribeController extends Controller
{
    /**
     * Store a new newsletter subscriber with tracking.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'email'        => 'required|email|max:190',
            'name'         => 'nullable|string|max:45',

            // account may be numeric (table is BIGINT unsigned)
            'account'      => 'nullable|string',
            'platform'     => 'nullable|string|max:22',

            'ip'           => 'nullable|string|max:80',
            'lat'          => 'nullable|string|max:80',
            'lng'          => 'nullable|string|max:80',
            'country'      => 'nullable|string|max:80',
            'city'         => 'nullable|string|max:80',
            'region'       => 'nullable|string|max:80',

            // UTM/Attribution
            'source'     => 'nullable|string|max:255',
            'medium'     => 'nullable|string|max:255',
            'campaign'   => 'nullable|string|max:255',
            'term'       => 'nullable|string|max:255',
            'content'    => 'nullable|string|max:255',
            'referrer'   => 'nullable|string|max:512',
            'page_url'   => 'nullable|string|max:512',
            'visitor_id' => 'nullable|string|max:64',
        ]);

        $merged = TrackingService::collect($request, $validated);

        $payload = [
            'email'   => $merged['email'],
            'name'    => $merged['name'] ?? null,

            // cast account if numeric, else null (table is BIGINT)
            'account' => \App\Services\TrackingService::castAccountId($merged['account'] ?? null),
            'platform'=> $merged['platform'] ?? null,

            // ip & geo mapping
            'ip_1'    => $merged['ip'] ?? null,
            'ip_2'    => $merged['platform_ip_2'] ?? null,
            'lat'     => $merged['lat'] ?? null,
            'lng'     => $merged['lng'] ?? null,
            'country' => $merged['country'] ?? null,
            'city'    => $merged['city'] ?? null,
            'region'  => $merged['region'] ?? null,

            // attribution
            'source'     => $merged['source'] ?? null,
            'medium'     => $merged['medium'] ?? null,
            'campaign'   => $merged['campaign'] ?? null,
            'term'       => $merged['term'] ?? null,
            'content'    => $merged['content'] ?? null,
            'referrer'   => $merged['referrer'] ?? null,
            'page_url'   => $merged['page_url'] ?? null,
            'visitor_id' => $merged['visitor_id'] ?? null,
        ];

        NewsletterSubscriber::create($payload);

        return response()->json([
            'success' => true,
            'message' => 'Subscription stored with tracking.',
        ]);
    }
}
