<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\Department\StoreDepartmentRequest;
use App\Http\Requests\Department\UpdateDepartmentRequest;
use App\Http\Resources\DepartmentResource;
use App\Models\Department;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class DepartmentController extends Controller
{
    public function __construct()
    {
        $this->middleware('web');
        $this->middleware('auth');
    }

    public function index(): AnonymousResourceCollection
    {
        $departments = Department::with(['parent', 'children', 'teamMembers'])
            ->whereNull('parent_id')
            ->get();

        return DepartmentResource::collection($departments);
    }

    public function store(StoreDepartmentRequest $request): DepartmentResource
    {
        $department = Department::create($request->validated());
        return new DepartmentResource($department);
    }

    public function show(Department $department): DepartmentResource
    {
        $department->load(['children', 'teamMembers']);
        return new DepartmentResource($department);
    }

    public function update(UpdateDepartmentRequest $request, Department $department): DepartmentResource
    {
        $department->update($request->validated());
        return new DepartmentResource($department);
    }

    public function destroy(Department $department)
    {
        // Check if department has no children before deletion
        if ($department->children()->exists()) {
            return response()->json([
                'message' => 'Cannot delete department with child departments'
            ], 422);
        }

        $department->delete();
        return response()->json(['message' => 'Department deleted successfully']);
    }
}
