<?php

namespace App\Http\Controllers\BRNC;

use App\Http\Controllers\Controller;
use App\Models\LandingFormTracking;
use App\Services\TrackingService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

/**
 * Class ContactController
 *
 * Handles public contact form submissions and persists full tracking data.
 * Maps incoming request keys to landing_form_tracking schema.
 */
class ContactController extends Controller
{
    /**
     * Store a new contact form submission with tracking data.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request): JsonResponse
    {
        // 1) Validate user fields (+ optional tracking)
        $validated = $request->validate([
            'name'          => 'required|string|max:44',
            'mobile'        => 'required|string|max:21',
            'services'      => 'required',
            'address'       => 'nullable|string',
            'services_for'  => 'nullable|string|max:44',

            // flexible inputs (we map them to schema fields below)
            'account'       => 'nullable|string|max:21',
            'account_name'  => 'nullable|string|max:21',
            'platform'      => 'nullable|string|max:21',
            'platform_info' => 'nullable|string|max:21',
            'ip'            => 'nullable|string|max:41',
            'lat'           => 'nullable|string|max:21',
            'lng'           => 'nullable|string|max:21',
            'country'       => 'nullable|string|max:21',
            'city'          => 'nullable|string|max:21',
            'region'        => 'nullable|string|max:44',

            // UTM/Attribution (added by your migration)
            'source'     => 'nullable|string|max:255',
            'medium'     => 'nullable|string|max:255',
            'campaign'   => 'nullable|string|max:255',
            'term'       => 'nullable|string|max:255',
            'content'    => 'nullable|string|max:255',
            'referrer'   => 'nullable|string|max:512',
            'page_url'   => 'nullable|string|max:512',
            'visitor_id' => 'nullable|string|max:64',

            // Optional email
            'email'        => 'nullable|string|max:190',
            // Optional tracking code/voucher if you use them
            'tracking_code'=> 'nullable|string|max:44',
            'voucher_id'   => 'nullable|string|max:22',
        ]);

        // 2) Merge with common tracking (adds platform_ip_2, source, referrer, page_url, visitor_id...)
        $merged = TrackingService::collect($request, $validated);

        // 3) Map inputs to table column names (landing_form_tracking schema)
        $payload = [
            'name'            => $merged['name'],
            'mobile'          => $merged['mobile'],
            'address'         => $merged['address'] ?? '--',
            'services'        => $merged['services'],
            'services_for'    => $merged['services_for'] ?? 'The Same Account',

            // account/platform mapping
            'account_name'    => $merged['account_name'] ?? ($merged['account'] ?? null),
            'platform_info'   => $merged['platform_info'] ?? ($merged['platform'] ?? null),

            // IP & geo mapping
            'platform_ip_1'   => $merged['ip'] ?? null,
            'platform_ip_2'   => $merged['platform_ip_2'] ?? null,
            'ip_lat'          => $merged['lat'] ?? null,
            'ip_lng'          => $merged['lng'] ?? null,
            'ip_country'      => $merged['country'] ?? null,
            'ip_city'         => $merged['city'] ?? null,
            'ip_region'       => $merged['region'] ?? null,

            // optional codes
            'tracking_code'   => $merged['tracking_code'] ?? '0',
            'voucher_id'      => $merged['voucher_id'] ?? '0',

            // attribution fields (from your new migration)
            'source'          => $merged['source'] ?? null,
            'medium'          => $merged['medium'] ?? null,
            'campaign'        => $merged['campaign'] ?? null,
            'term'            => $merged['term'] ?? null,
            'content'         => $merged['content'] ?? null,
            'referrer'        => $merged['referrer'] ?? null,
            'page_url'        => $merged['page_url'] ?? null,
            'visitor_id'      => $merged['visitor_id'] ?? null,

            // optional email (your old controller used '--' fallback)
            'email'           => $merged['email'] ?? ($request->input('email', '--')),
        ];

        LandingFormTracking::create($payload);

        return response()->json([
            'success' => true,
            'message' => 'Contact form submitted with tracking.',
        ]);
    }
}
