<?php

namespace App\Http\Controllers\BRNC;

use App\Http\Controllers\Controller;
use App\Models\BtnTrack;
use App\Models\Contacts;
use App\Services\TrackingService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

/**
 * Class ServerController
 *
 * 1) contact(): returns account/platform contact meta (ext, phones, social).
 * 2) track(): generic button click tracker (WhatsApp / Phone Call / other)
 *    with full attribution fields persisted to btn_track.
 */
class ServerController extends Controller
{
    /**
     * Fetch contact info by account & platform.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function contact(Request $request): JsonResponse
    {
        $payload = $request->validate([
            'account'  => 'required|string|max:24',
            'platform' => 'required|string|max:24',
        ]);

        $contact = Contacts::where('account', $payload['account'])
            ->where('platform', $payload['platform'])
            ->first([
                'ext', 'phone_call', 'whats_call',
                'email', 'Instagram', 'facebook', 'Snapchat', 'Tiktok', 'youtube'
            ]);

        if (!$contact) {
            $contact = [
                'ext'        => '965',
                'phone_call' => '12345678',
                'whats_call' => '12345678',
                'email'      => 'info@example.com',
                'Instagram'  => '#',
                'facebook'   => '#',
                'Snapchat'   => '#',
                'Tiktok'     => '#',
                'youtube'    => '#',
            ];
        }

        return response()->json($contact);
    }

    /**
     * Track a button click (WhatsApp / Phone Call / Custom) with attribution.
     *
     * Table: btn_track (account, platform, ip_1, ip_2, click_type, lat, lng, country, city, region, + UTM fields)
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function track(Request $request): JsonResponse
    {
        // Flexible: allow "click_type" directly OR old "insert" flag (0=>WhatsApp, else Phone)
        $validated = $request->validate([
            'account'    => 'required|string|max:24',
            'platform'   => 'required|string|max:24',
            'ip'         => 'nullable|string|max:24', // will map to ip_1
            'lat'        => 'nullable|string|max:24',
            'lng'        => 'nullable|string|max:24',
            'country'    => 'nullable|string|max:24',
            'city'       => 'nullable|string|max:24',
            'region'     => 'nullable|string|max:24',
            'click_type' => 'nullable|string|max:24',
            'insert'     => 'nullable|in:0,1',

            // UTM/Attribution
            'source'     => 'nullable|string|max:255',
            'medium'     => 'nullable|string|max:255',
            'campaign'   => 'nullable|string|max:255',
            'term'       => 'nullable|string|max:255',
            'content'    => 'nullable|string|max:255',
            'referrer'   => 'nullable|string|max:512',
            'page_url'   => 'nullable|string|max:512',
            'visitor_id' => 'nullable|string|max:64',
        ]);

        $merged = TrackingService::collect($request, $validated);

        // derive click type (backward compatible)
        $clickType = $merged['click_type']
            ?? (($request->input('insert') === "0" || $request->input('insert') === 0) ? 'WhatsApp' : 'Phone Call');

        $payload = [
            'account'   => $merged['account'],
            'platform'  => $merged['platform'],
            'ip_1'      => $merged['ip'] ?? null,
            'ip_2'      => $merged['platform_ip_2'] ?? null,
            'click_type'=> $clickType,
            'lat'       => $merged['lat'] ?? null,
            'lng'       => $merged['lng'] ?? null,
            'country'   => $merged['country'] ?? null,
            'city'      => $merged['city'] ?? null,
            'region'    => $merged['region'] ?? null,

            // attribution
            'source'     => $merged['source'] ?? null,
            'medium'     => $merged['medium'] ?? null,
            'campaign'   => $merged['campaign'] ?? null,
            'term'       => $merged['term'] ?? null,
            'content'    => $merged['content'] ?? null,
            'referrer'   => $merged['referrer'] ?? null,
            'page_url'   => $merged['page_url'] ?? null,
            'visitor_id' => $merged['visitor_id'] ?? null,
        ];

        BtnTrack::create($payload);

        return response()->json([
            'success' => true,
            'message' => 'Click tracked.',
        ]);
    }
}
