<?php

namespace App\Http\Controllers;

use App\Models\Department;
use App\Models\Employee;
use App\Models\AttendanceLog;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Carbon\Carbon;

class DepartmentController extends Controller
{
    public function index(Request $request)
    {
        $departments = Department::with(['employees.user', 'parent'])
            ->withCount('employees')
            ->when(method_exists(Department::class, 'scopeActive'), fn($q) => $q->active())
            ->orderBy('name')
            ->get();

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => $departments->map(fn($d) => [
                    'id'               => $d->id,
                    'name'             => $d->name,
                    'code'             => $d->code,
                    'parent_id'        => $d->parent_id,
                    'color_code'       => $d->color_code,
                    'is_active'        => (bool) $d->is_active,
                    'employees_count'  => (int) $d->employees_count,
                ]),
            ]);
        }

        return view('pages.departments.index', compact('departments'));
    }

    public function show(Request $request, Department $department)
    {
        $department->load(['employees.user', 'parent', 'children'])->loadCount('employees');

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => [
                    'id'              => $department->id,
                    'name'            => $department->name,
                    'code'            => $department->code,
                    'parent_id'       => $department->parent_id,
                    'color_code'      => $department->color_code,
                    'is_active'       => (bool) $department->is_active,
                    'employees_count' => (int) $department->employees_count,
                    'children'        => $department->children->map(fn($c)=>[
                        'id'=>$c->id,'name'=>$c->name,'code'=>$c->code
                    ]),
                    'employees'        => $department->employees->map(fn($e)=>[
                        'id'=>$e->id,
                        'employee_code'=>$e->employee_code,
                        'name'=>$e->full_name ?? trim(($e->first_name ?? '').' '.($e->last_name ?? '')),
                        'email'=>optional($e->user)->email,
                        'status'=>$e->status,
                        'hire_date'=>optional($e->hire_date)?->format('Y-m-d'),
                    ]),
                ],
            ]);
        }

        return view('pages.departments.show', compact('department'));
    }

    public function employees($departmentId)
    {
        $department = Department::with(['employees.user'])->findOrFail($departmentId);

        return response()->json([
            'success'    => true,
            'department' => $department->name,
            'employees'  => $department->employees->map(function ($employee) {
                return [
                    'id'            => $employee->id,
                    'employee_code' => $employee->employee_code,
                    'first_name'    => $employee->first_name,
                    'last_name'     => $employee->last_name,
                    'name'          => $employee->full_name ?? trim(($employee->first_name ?? '').' '.($employee->last_name ?? '')),
                    'user_id'       => $employee->user_id,
                    'email'         => optional($employee->user)->email,
                    'department_id' => $employee->department_id,
                    'status'        => $employee->status,
                    'hire_date'     => optional($employee->hire_date)?->format('Y-m-d'),
                ];
            }),
        ]);
    }

    public function attendance($departmentId, Request $request)
    {
        // تحقّق من التاريخ (لو اتبعت غلط يطلع 422 Validation)
        $request->validate([
            'date' => ['nullable','date_format:Y-m-d'],
        ]);

        $date = $request->get('date', now()->format('Y-m-d'));
        $department = Department::findOrFail($departmentId);

        $employees = $department->employees()->with('user')->get();
        $employeeCodes = $employees->pluck('employee_code')->filter()->values();

        $attendanceLogs = AttendanceLog::whereIn('employee_code', $employeeCodes)
            ->whereDate('check_time', $date)
            ->orderBy('check_time')
            ->get()
            ->groupBy('employee_code');

        $attendanceData = [];

        foreach ($employees as $employee) {
            $logs = $attendanceLogs->get($employee->employee_code, collect());
            $checkIn  = $logs->where('type', 'check_in')->first();
            $checkOut = $logs->where('type', 'check_out')->last();

            if ($checkIn || $checkOut) {
                $attendanceData[] = [
                    'employee'    => [
                        'name' => $employee->full_name ?? trim(($employee->first_name ?? '').' '.($employee->last_name ?? '')),
                        'code' => $employee->employee_code,
                    ],
                    'check_in'    => optional($checkIn?->check_time)?->format('H:i:s'),
                    'check_out'   => optional($checkOut?->check_time)?->format('H:i:s'),
                    'total_hours' => $this->calculateTotalHours($checkIn, $checkOut),
                ];
            }
        }

        return response()->json([
            'success'    => true,
            'department' => $department->name,
            'date'       => $date,
            'attendance' => $attendanceData,
        ]);
    }

    private function calculateTotalHours($checkIn, $checkOut)
    {
        if (!$checkIn || !$checkOut) {
            return 'In Progress';
        }

        $start = Carbon::parse($checkIn->check_time);
        $end   = Carbon::parse($checkOut->check_time);

        return number_format($end->diffInMinutes($start) / 60, 2) . ' hours';
    }

    public function employeeAttendance($employeeCode, Request $request)
    {
        $request->validate([
            'date' => ['nullable','date_format:Y-m-d'],
        ]);

        $date = $request->get('date', now()->format('Y-m-d'));

        $employee = Employee::where('employee_code', $employeeCode)->first();

        if (!$employee) {
            return response()->json([
                'success' => false,
                'message' => 'Employee not found.',
            ], 404);
        }

        $attendanceLogs = AttendanceLog::where('employee_code', $employeeCode)
            ->whereDate('check_time', $date)
            ->orderBy('check_time')
            ->get();

        $checkIns = $attendanceLogs->where('type', 'check_in');
        $checkOuts = $attendanceLogs->where('type', 'check_out');

        $attendanceData = [];
        $totalHours = 0;

        foreach ($checkIns as $index => $checkIn) {
            $checkOut = $checkOuts->skip($index)->first();

            $record = [
                'check_in' => $checkIn->check_time->format('H:i:s'),
                'check_out' => $checkOut ? $checkOut->check_time->format('H:i:s') : null,
                'duration' => null,
                'status' => $checkOut ? 'Complete' : 'In Progress'
            ];

            if ($checkOut) {
                $start = Carbon::parse($checkIn->check_time);
                $end = Carbon::parse($checkOut->check_time);
                $hours = $end->diffInMinutes($start) / 60;
                $totalHours += $hours;
                $record['duration'] = number_format($hours, 2) . ' hours';
            }

            $attendanceData[] = $record;
        }

        return response()->json([
            'success' => true,
            'employee' => [
                'code' => $employee->employee_code,
                'name' => $employee->full_name ?? trim(($employee->first_name ?? '').' '.($employee->last_name ?? '')),
                'department' => $employee->department ? $employee->department->name : 'N/A'
            ],
            'date' => $date,
            'attendance' => $attendanceData,
            'summary' => [
                'total_hours' => number_format($totalHours, 2) . ' hours',
                'check_ins' => $checkIns->count(),
                'check_outs' => $checkOuts->count(),
                'status' => $checkIns->count() === $checkOuts->count() ? 'Complete' : 'In Progress'
            ]
        ]);
    }

    public function store(Request $request)
    {
        // Validation يطلّع 422 {message, errors} تلقائيًا للـ AJAX
        $validated = $request->validate([
            'name'       => ['required','string','max:255'],
            'code'       => ['required','string','max:50','unique:departments,code'],
            'color_code' => ['required','string','max:7'],
            'parent_id'  => ['nullable','exists:departments,id'],
            'is_active'  => ['nullable','boolean'],
        ]);

        // منع parent_id = نفس القسم (لو حد لعب في الفورم)
        if (!empty($validated['parent_id']) && isset($request->id) && (int)$validated['parent_id'] === (int)$request->id) {
            return $request->expectsJson()
                ? response()->json(['success'=>false,'message'=>'A department cannot be its own parent.'], 422)
                : back()->with('error','A department cannot be its own parent.');
        }

        $department = Department::create([
            'name'       => $validated['name'],
            'code'       => $validated['code'],
            'color_code' => $validated['color_code'],
            'parent_id'  => $validated['parent_id'] ?? null,
            'is_active'  => (bool)($validated['is_active'] ?? true),
        ]);

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Department created successfully.',
                'data'    => ['id' => $department->id],
            ], 201);
        }

        return redirect()->route('departments.index')
            ->with('success', 'Department created successfully.');
    }

    public function update(Request $request, Department $department)
    {
        // لو هتسمح بتعديل code: أضف Rule unique مع ignore
        $validated = $request->validate([
            'name'       => ['required','string','max:255'],
            'color_code' => ['required','string','max:7'],
            'parent_id'  => ['nullable','exists:departments,id'],
            'code'       => ['sometimes','string','max:50', Rule::unique('departments','code')->ignore($department->id)],
            'is_active'  => ['sometimes','boolean'],
        ]);

        if (!empty($validated['parent_id']) && (int)$validated['parent_id'] === (int)$department->id) {
            return $request->expectsJson()
                ? response()->json(['success'=>false,'message'=>'A department cannot be its own parent.'], 422)
                : back()->with('error','A department cannot be its own parent.');
        }

        $department->update([
            'name'       => $validated['name'],
            'color_code' => $validated['color_code'],
            'parent_id'  => $validated['parent_id'] ?? null,
            'code'       => $validated['code'] ?? $department->code,
            'is_active'  => array_key_exists('is_active',$validated) ? (bool)$validated['is_active'] : $department->is_active,
        ]);

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Department updated successfully.',
                'data'    => ['id' => $department->id],
            ]);
        }

        return back()->with('success', 'Department updated successfully');
    }

    public function destroy(Request $request, Department $department)
    {
        if ($department->children()->exists()) {
            $msg = 'Cannot delete department with child departments.';
            return $request->expectsJson()
                ? response()->json(['success'=>false,'message'=>$msg], 409)
                : back()->with('error', $msg);
        }

        if ($department->employees()->exists()) {
            $msg = 'Cannot delete department with employees.';
            return $request->expectsJson()
                ? response()->json(['success'=>false,'message'=>$msg], 409)
                : back()->with('error', $msg);
        }

        $department->delete();

        if ($request->expectsJson()) {
            return response()->json(['success'=>true,'message'=>'Department deleted.']);
        }

        return back()->with('success', 'Department deleted successfully');
    }
}
