<?php
declare(strict_types=1);

namespace App\Http\Controllers;

use App\Services\EmailService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Response;
use Mews\Purifier\Facades\Purifier;
use Carbon\Carbon;
use App\Support\MailFmt;

/**
 * Class EmailController
 *
 * Handles message-centric actions:
 * - List inbox messages (with basic pagination).
 * - Show message details by UID (HTML sanitized).
 * - Download attachments by index.
 * - Mark messages read/unread.
 * - List folders & search.
 * - (Optional) compose & send via per-user SMTP.
 *
 * @package App\Http\Controllers
 *
 * @see \App\Services\EmailService
 */
class EmailController extends Controller
{
    /**
     * @var EmailService
     */
    private EmailService $mail;

    /**
     * EmailController constructor.
     *
     * Applies "auth" middleware and injects EmailService.
     *
     * @param  EmailService  $mail
     */
    public function __construct(EmailService $mail)
    {
        $this->middleware('auth');
        $this->mail = $mail;
    }

    /**
     * GET /email/inbox
     *
     * Render inbox with basic pagination and optional folder selection.
     *
     * @param  Request  $request
     * @return \Illuminate\Contracts\View\View
     */
    public function inbox(Request $request)
    {
        $page   = max(1, (int) $request->integer('page', 1));
        $limit  = min(100, max(5, (int) $request->integer('limit', 25)));
        $folder = $request->query('folder');

        $result = $this->mail->listInbox(page: $page, limit: $limit, folder: $folder);

        return view('email.inbox', [
            'items'      => $result['data'],
            'pagination' => $result['pagination'],
            'folder'     => $result['folder'],
        ]);
    }

    /**
     * GET /email/detail/{uid}
     *
     * Show a single message by UID; HTML is sanitized before rendering.
     *
     * @param  string   $uid
     * @param  Request  $request
     * @return \Illuminate\Contracts\View\View|\Symfony\Component\HttpFoundation\Response
     */
    // public function detail(string $uid, Request $request)
    // {
    //     $folder = $request->query('folder');
    //     $msg    = $this->mail->getMessageByUID($uid, $folder);
    //     abort_unless($msg, 404);

    //     // عناوين البريد (تحويل آمن)
    //     $from = MailFmt::addressesToString($msg->getFrom());
    //     $to   = MailFmt::addressesToString($msg->getTo());
    //     $cc   = MailFmt::addressesToString($msg->getCc());
    //     $bcc  = MailFmt::addressesToString($msg->getBcc());

    //     // الموضوع + التاريخ + حالة القراءة
    //     $subject = MailFmt::string($msg->getSubject()) ?? '(No subject)';
    //     $date    = MailFmt::date($msg->getDate());
    //     $seen    = MailFmt::isSeen($msg);

    //     // الأجسام (مع تعقيم الـ HTML)
    //     $html = $msg->getHTMLBody();
    //     $text = $msg->getTextBody();
    //     $safeHtml = $html ? Purifier::clean($html, 'default') : null;

    //     // المرفقات (ميتاداتا من الخدمة)
    //     $attachments = $this->mail->getAttachmentsMeta($uid, $folder);

    //     return view('email.detail', [
    //         'uid'         => $uid,
    //         'folder'      => $folder ?: 'INBOX',
    //         'subject'     => $subject,
    //         'from'        => $from,
    //         'to'          => $to,
    //         'cc'          => $cc,
    //         'bcc'         => $bcc,
    //         'date'        => $date,
    //         'safeHtml'    => $safeHtml,
    //         'text'        => $text,
    //         'attachments' => $attachments,
    //         'seen'        => $seen,
    //     ]);
    // }

    public function detail(string $uid, Request $request)
    {
        $folder = $request->query('folder');

        // === قائمة جانبية (mini inbox) ===
        $sp = max(1, (int) $request->integer('sp', 1));       // sidebar page
        $sl = min(100, max(5, (int) $request->integer('sl', 30))); // sidebar limit
        $side = $this->mail->listInbox(page: $sp, limit: $sl, folder: $folder);

        // === الرسالة المختارة ===
        $msg = $this->mail->getMessageByUID($uid, $folder);
        abort_unless($msg, 404);

        $from    = MailFmt::addressesToString($msg->getFrom());
        $to      = MailFmt::addressesToString($msg->getTo());
        $cc      = MailFmt::addressesToString($msg->getCc());
        $bcc     = MailFmt::addressesToString($msg->getBcc());
        $subject = MailFmt::string($msg->getSubject()) ?? '(No subject)';
        $date    = MailFmt::date($msg->getDate());
        $seen    = MailFmt::isSeen($msg);

        $html = $msg->getHTMLBody();
        $text = $msg->getTextBody();
        $safeHtml = $html ? Purifier::clean($html, 'default') : null;

        $attachments = $this->mail->getAttachmentsMeta($uid, $folder);

        return view('email.detail', [
            // detail payload
            'uid'         => $uid,
            'folder'      => $folder ?: 'INBOX',
            'subject'     => $subject,
            'from'        => $from,
            'to'          => $to,
            'cc'          => $cc,
            'bcc'         => $bcc,
            'date'        => $date,
            'safeHtml'    => $safeHtml,
            'text'        => $text,
            'attachments' => $attachments,
            'seen'        => $seen,

            // sidebar payload
            'sideItems'       => $side['data'],
            'sidePagination'  => $side['pagination'],
            'sideFolder'      => $side['folder'],
            'sideActiveUid'   => $uid,
        ]);
    }




    /**
     * GET /email/attachment/{uid}/{index?}
     *
     * Stream an attachment by its 0-based index in the message.
     *
     * @param  string   $uid
     * @param  int      $index
     * @param  Request  $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function attachment(string $uid, int $index = 0, Request $request)
    {
        $folder = $request->query('folder');
        $msg    = $this->mail->getMessageByUID($uid, $folder);
        abort_unless($msg, 404);

        $atts = array_values(iterator_to_array($msg->getAttachments() ?? []));
        abort_unless(isset($atts[$index]), 404);

        $att     = $atts[$index];
        $name    = $att->getName() ?: ('attachment-' . $index);
        $mime    = $att->getMimeType() ?: 'application/octet-stream';
        $content = $att->getContent();

        return Response::make($content, 200, [
            'Content-Type'        => $mime,
            'Content-Disposition' => 'attachment; filename="' . addslashes($name) . '"',
        ]);
    }

    /**
     * POST /email/mark-read
     *
     * Mark a message as read by UID.
     *
     * @param  Request  $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
    public function markRead(Request $request)
    {
        $data = $request->validate([
            'uid'    => ['required', 'string'],
            'folder' => ['nullable', 'string'],
        ]);

        $ok = $this->mail->setSeen($data['uid'], true, $data['folder'] ?? null);

        if ($request->wantsJson()) {
            return response()->json(['ok' => (bool) $ok]);
        }
        return back()->with($ok ? 'ok' : 'error', $ok ? 'Marked as read.' : 'Failed to mark as read.');
    }

    /**
     * POST /email/mark-unread
     *
     * Mark a message as unread by UID.
     *
     * @param  Request  $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
    public function markUnread(Request $request)
    {
        $data = $request->validate([
            'uid'    => ['required', 'string'],
            'folder' => ['nullable', 'string'],
        ]);

        $ok = $this->mail->setSeen($data['uid'], false, $data['folder'] ?? null);

        if ($request->wantsJson()) {
            return response()->json(['ok' => (bool) $ok]);
        }
        return back()->with($ok ? 'ok' : 'error', $ok ? 'Marked as unread.' : 'Failed to mark as unread.');
    }

    /**
     * GET /email/folders
     *
     * List available folders/mailboxes.
     *
     * @param  Request  $request
     * @return \Illuminate\Contracts\View\View|\Illuminate\Http\JsonResponse
     */
    public function folders(Request $request)
    {
        $folders = $this->mail->listFolders();

        if ($request->wantsJson()) {
            return response()->json($folders);
        }
        return view('email.folders', ['folders' => $folders]);
    }

    /**
     * GET /email/search
     *
     * Perform a basic search in the selected folder.
     * Supported query params: q, from, since (Y-m-d), before (Y-m-d), folder, page, limit
     *
     * @param  Request  $request
     * @return \Illuminate\Contracts\View\View
     */
    public function search(Request $request)
    {
        $page   = max(1, (int) $request->integer('page', 1));
        $limit  = min(100, max(5, (int) $request->integer('limit', 25)));
        $folder = $request->query('folder');

        $params = $request->validate([
            'q'      => ['nullable','string'],
            'from'   => ['nullable','string'],
            'since'  => ['nullable','date'],
            'before' => ['nullable','date'],
        ]);

        $result = $this->mail->search($params, $page, $limit, $folder);

        return view('email.inbox', [
            'items'      => $result['data'],
            'pagination' => $result['pagination'],
            'folder'     => $folder ?: 'INBOX',
            'query'      => $params,
            'searching'  => true,
        ]);
    }

    /**
     * GET /email/compose
     *
     * Render compose view (optional).
     *
     * @return \Illuminate\Contracts\View\View
     */
   public function compose(Request $request)
{
    // يسمح بتمرير قيم ابتدائية عبر ?to=&subject=&body=
    return view('email.compose', [
        'init' => [
            'to'      => (string) $request->query('to', ''),
            'cc'      => (string) $request->query('cc', ''),
            'bcc'     => (string) $request->query('bcc', ''),
            'subject' => (string) $request->query('subject', ''),
            'body'    => (string) $request->query('body', ''),
        ],
    ]);
}

/** Send via per-user SMTP (يدعم مرفقات اختيارية) */
public function send(Request $request)
{
    $data = $request->validate([
        'to'           => ['required','string'],
        'subject'      => ['required','string'],
        'body'         => ['required','string'],
        'cc'           => ['nullable','string'],
        'bcc'          => ['nullable','string'],
        'attachments'  => ['nullable','array'],
        'attachments.*'=> ['file','max:10240'], // 10MB لكل ملف
    ]);

    // parser مرن: comma / semicolon / whitespace
    $split = function (?string $s): array {
        if (!$s) return [];
        $s = str_replace([';', "\n", "\r", "\t"], ',', $s);
        $parts = array_filter(array_map('trim', explode(',', $s)));
        // تصفية إيميلات فقط
        return array_values(array_filter($parts, fn($e) => filter_var($e, FILTER_VALIDATE_EMAIL)));
    };

    $to  = $split($data['to']);
    $cc  = $split($data['cc'] ?? null) ?: null;
    $bcc = $split($data['bcc'] ?? null) ?: null;

    if (empty($to)) {
        return back()->withInput()->withErrors(['to' => 'Please provide at least one valid email address.']);
    }

    // مرفقات (اختياري)
    $files = $request->file('attachments', []) ?: [];

    // لو فعلت دعم المرفقات في EmailService (انظر القسم 3)
    if (method_exists($this->mail, 'sendEmail')) {
        $ref = new \ReflectionMethod($this->mail, 'sendEmail');
        if ($ref->getNumberOfParameters() >= 5) {
            // signature تدعم $attachments كآخر بارامتر
            $this->mail->sendEmail($to, $data['subject'], $data['body'], $cc, $bcc, $files);
        } else {
            // بدون مرفقات
            $this->mail->sendEmail($to, $data['subject'], $data['body'], $cc, $bcc);
        }
    }

    return redirect()->route('email.inbox')->with('ok', 'Email sent successfully.');
}

}
