<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Database\QueryException;
use App\Http\Requests\Employee\StoreEmployeeRequest;
use App\Http\Requests\Employee\UpdateEmployeeRequest;
use App\Models\Employee;

class EmployeeController extends Controller
{
    /**
     * Create a new employee.
     * - AJAX: JSON { success, message, data:{id} } + 201
     * - Web:  redirect back with success flash
     */
    public function store(StoreEmployeeRequest $request)
    {
        $this->authorize('create', Employee::class);
        $emp = Employee::create($request->validated());

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Employee created successfully.',
                'data'    => ['id' => $emp->id],
            ], 201);
        }

        return back()->with('success', 'Employee created successfully');
    }

    /**
     * Update an existing employee.
     * - AJAX: JSON { success, message, data:{id} }
     * - Web:  redirect back with success flash
     */
    public function update(UpdateEmployeeRequest $request, Employee $employee)
    {
        $this->authorize('update', $employee);
        $employee->update($request->validated());

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Employee updated successfully.',
                'data'    => ['id' => $employee->id],
            ]);
        }

        return back()->with('success', 'Employee updated successfully');
    }

    /**
     * Delete an employee.
     * - لو في علاقات مانعة للحذف (FK) نرجّع 409 برسالة واضحة.
     * - AJAX: JSON { success, message }
     * - Web:  redirect back with flash
     */
    public function destroy(Request $request, Employee $employee)
    {
        $this->authorize('delete', $employee);
        try {
            $employee->delete();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Employee deleted.',
                ]);
            }

            return back()->with('success', 'Employee deleted successfully');
        } catch (QueryException $e) {
            // مثال: attendance logs, memberships, ... إلخ
            $msg = 'Cannot delete employee due to related records.';

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $msg,
                ], 409);
            }

            return back()->with('error', $msg);
        }
    }
    public function show(Request $request, Employee $employee)
    {
        $this->authorize('view', $employee);
        // علاقات أساسية للمودال
        $employee->loadMissing([
            'user:id,name,email',
            'department:id,name',
        ]);

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'data'    => [
                    'id'             => $employee->id,
                    'employee_code'  => $employee->employee_code,
                    'first_name'     => $employee->first_name,
                    'last_name'      => $employee->last_name,
                    'name'           => $employee->full_name ?? trim(($employee->first_name ?? '') . ' ' . ($employee->last_name ?? '')),
                    'user_id'        => $employee->user_id,
                    'user_email'     => optional($employee->user)->email,
                    'department_id'  => $employee->department_id,
                    'department'     => optional($employee->department)->name,
                    'status'         => $employee->status,
                    'hire_date'      => optional($employee->hire_date)?->format('Y-m-d'),
                    'created_at'     => optional($employee->created_at)?->toIso8601String(),
                    'updated_at'     => optional($employee->updated_at)?->toIso8601String(),
                ]
            ]);
        }

        // في حالة طلب ويب عادي (نادرًا هنحتاجه)
        return view('pages.employees.show', compact('employee'));
    }
}
