<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use Google\Client;
use Google\Service\Sheets;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Log;

class GoogleSheetController extends Controller
{
    /**
     * Map of branch IDs to their names and corresponding Google Sheet names
     *
     * @var array
     */
    private $branchMap = [
        101 => ['name' => 'القادسية 1', 'sheet' => 'Name Q1'],
        102 => ['name' => 'القادسية 2', 'sheet' => 'Name Q2'],
        104 => ['name' => 'القادسية 3 (المطابخ)', 'sheet' => 'Name Q3'],
        103 => ['name' => 'مدينة نصر', 'sheet' => 'Name N2'],
        105 => ['name' => 'فرع الكويت', 'sheet' => 'name kw1'],
    ];

    /**
     * Import attendance data from Google Sheets
     *
     * @return string
     */
    public function importAttendance()
    {
        set_time_limit(120); // Increase execution time to 120 seconds

        $client = new Client();
        $client->setApplicationName('Attendance Integration');
        $client->setScopes(Sheets::SPREADSHEETS_READONLY);
        $client->setAuthConfig(config('app.google_credentials'));

        $service = new Sheets($client);
        $spreadsheetId = '1UKZs3DrT11AkK8XI34SgJGDXVUFH-8yTGgxV64b0cts';

        $importState = DB::table('import_states')
            ->where('source_key', 'final_sheet')
            ->first();

        $lastRow = $importState ? $importState->last_row : 1;
        $buffer = 500;
        $startRow = max(1, $lastRow + 1 - $buffer);

        $nameMaps = [];
        foreach ($this->branchMap as $branchId => $info) {
            $nameSheet = $info['sheet'];
            $nameRange = "$nameSheet!A:B";
            $nameResponse = $service->spreadsheets_values->get($spreadsheetId, $nameRange, ['valueRenderOption' => 'FORMATTED_VALUE']);
            $nameValues = $nameResponse->getValues();
            array_shift($nameValues);

            $nameMaps[$branchId] = [];
            foreach ($nameValues as $nameRow) {
                if (isset($nameRow[1])) {
                    $nameMaps[$branchId][$nameRow[1]] = $nameRow[0] ?? 'Unknown';
                }
            }
        }

        $response = $service->spreadsheets_values->get($spreadsheetId, 'Final!A1:D', ['valueRenderOption' => 'FORMATTED_VALUE']);
        $allValues = $response->getValues();
        $totalRows = count($allValues) - 1;

        $batchSize = 500;
        $processed = 0;
        $unknownCount = 0;

        for ($currentRow = $startRow; $currentRow <= $totalRows; $currentRow += $batchSize) {
            $endRow = min($currentRow + $batchSize - 1, $totalRows);
            $range = "Final!A{$currentRow}:D{$endRow}";

            try {
                $response = $service->spreadsheets_values->get($spreadsheetId, $range, ['valueRenderOption' => 'FORMATTED_VALUE']);
                $values = $response->getValues();

                if (empty($values)) {
                    Log::info("No data found for range: $range");
                    continue;
                }

                $insertData = [];
                foreach ($values as $row) {
                    $correctId = $row[1] ?? null;
                    $checkTime = $row[2] ?? null;
                    $sensorId = (int) ($row[3] ?? 0);

                    if (!$correctId || !$checkTime || !array_key_exists($sensorId, $this->branchMap)) {
                        Log::warning("Skipped row: Missing data or invalid SensorID=$sensorId");
                        continue;
                    }

                    $branchInfo = $this->branchMap[$sensorId];
                    $employeeName = $nameMaps[$sensorId][$correctId] ?? 'Unknown';

                    if ($employeeName === 'Unknown') {
                        Log::warning("Unknown employee name for correctId=$correctId, sensorId=$sensorId");
                        $unknownCount++;
                    }

                    $insertData[] = [
                        'employee_code' => $correctId,
                        'check_time' => $checkTime,
                        'branch_id' => $sensorId,
                        'employee_name' => $employeeName,
                        'branch_name' => $branchInfo['name'],
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];

                    $processed++;
                }

                if (!empty($insertData)) {
                    DB::table('_x_test_attendances')->upsert(
                        $insertData,
                        ['employee_code', 'check_time', 'branch_id'],
                        ['employee_name', 'branch_name', 'created_at', 'updated_at']
                    );
                    Log::info("Inserted/Updated " . count($insertData) . " records");
                }
            } catch (\Exception $e) {
                Log::error("Error: " . $e->getMessage());
                sleep(1);
            }
        }

        if ($processed > 0) {
            DB::table('import_states')
                ->updateOrInsert(
                    ['source_key' => 'final_sheet'],
                    [
                        'last_row' => $totalRows,
                        'last_imported_at' => now(),
                        'meta' => json_encode(['processed' => $processed, 'unknown_count' => $unknownCount])
                    ]
                );
        }

        return "Imported $processed attendance records successfully! (Unknown names: $unknownCount)";
    }

    /**
     * Migrate data from x_test_attendances_x to attendance_logs
     *
     * @return string
     */
    // public function migrateToAttendanceLogs()
    // {
    //     set_time_limit(300); // Increase execution time to 300 seconds (5 minutes)

    //     $batchSize = 500; // Reduce batch size for better performance
    //     $total = DB::table('_x_test_attendances')->count();

    //     $insertedCount = 0;

    //     for ($offset = 0; $offset < $total; $offset += $batchSize) {
    //         $data = DB::table('_x_test_attendances')
    //             ->select('employee_code', 'check_time', 'branch_id', 'employee_name', 'branch_name')
    //             ->orderBy('check_time')
    //             ->offset($offset)
    //             ->limit($batchSize)
    //             ->get();

    //         $insertData = [];
    //         $skippedCount = 0;
    //         $skippedReasons = [];

    //         foreach ($data as $row) {
    //             $isValid = true;
    //             $reason = '';

    //             if (!$row->check_time) {
    //                 $isValid = false;
    //                 $reason = 'null/empty check_time';
    //             } elseif (!Carbon::hasFormat($row->check_time, 'Y-m-d H:i:s')) {
    //                 $isValid = false;
    //                 $reason = 'invalid check_time format';
    //             }

    //             if (!$isValid) {
    //                 Log::warning("Skipping row at offset $offset for employee_code: $row->employee_code, reason: $reason, data: " . json_encode($row));
    //                 $skippedCount++;
    //                 $skippedReasons[$reason] = ($skippedReasons[$reason] ?? 0) + 1;
    //                 continue;
    //             }

    //             $type = Carbon::parse($row->check_time)->hour < 14 ? 'check_in' : 'check_out';

    //             $insertData[] = [
    //                 'employee_code' => $row->employee_code,
    //                 'check_time' => $row->check_time,
    //                 'branch_id' => $row->branch_id,
    //                 'type' => $type,
    //                 'source' => 'device',
    //                 'employee_name' => $row->employee_name,
    //                 'branch_name' => $row->branch_name,
    //                 'created_at' => now(),
    //                 'updated_at' => now(),
    //             ];
    //         }

    //         Log::info("Processed batch at offset $offset, skipped $skippedCount rows (Reasons: " . json_encode($skippedReasons) . "), preparing to insert " . count($insertData) . " records");

    //         if (!empty($insertData)) {
    //             try {
    //                 $chunks = array_chunk($insertData, 100);
    //                 foreach ($chunks as $chunk) {
    //                     $inserted = DB::table('attendance_logs')->insert($chunk); // Use insert instead of insertOrIgnore
    //                     $insertedCount += $inserted;
    //                     Log::info("Migrated $inserted records to attendance_logs (offset: $offset, total inserted: $insertedCount)");
    //                 }
    //             } catch (\Exception $e) {
    //                 Log::error("Error migrating records at offset $offset: " . $e->getMessage());
    //             }
    //             $insertData = [];
    //         }
    //     }

    //     Log::info("Migration completed, total records inserted: $insertedCount out of $total");
    //     return "Migrated $insertedCount records to attendance_logs successfully!";
    // }


public function migrateToAttendanceLogs()
{
    set_time_limit(300); // زيادة الوقت لـ 300 ثانية (5 دقايق)

    $batchSize = 500; // حجم الدفعة لكل batch
    $total = DB::table('_x_test_attendances')->count();
    $insertedCount = 0;
    $skippedCount = 0;

    for ($offset = 0; $offset < $total; $offset += $batchSize) {
        $data = DB::table('_x_test_attendances')
            ->select('employee_code', 'check_time', 'branch_id', 'employee_name', 'branch_name')
            ->orderBy('check_time')
            ->offset($offset)
            ->limit($batchSize)
            ->get();

        $insertData = [];
        foreach ($data as $row) {
            if (!$row->check_time) {
                Log::warning("Skipping row with null/empty check_time for employee_code: $row->employee_code at offset $offset");
                $skippedCount++;
                continue;
            }

            $type = Carbon::parse($row->check_time)->hour < 14 ? 'check_in' : 'check_out';

            // التحقق من التكرار قبل الإضافة
            $existing = DB::table('attendance_logs')
                ->where('employee_code', $row->employee_code)
                ->where('check_time', $row->check_time)
                ->where('branch_id', $row->branch_id)
                ->exists();

            if ($existing) {
                Log::warning("Duplicate record skipped for employee_code: {$row->employee_code}, check_time: {$row->check_time}, branch_id: {$row->branch_id}");
                $skippedCount++;
                continue;
            }

            $insertData[] = [
                'employee_code' => $row->employee_code,
                'check_time' => $row->check_time,
                'branch_id' => $row->branch_id,
                'type' => $type,
                'source' => 'device',
                'employee_name' => $row->employee_name,
                'branch_name' => $row->branch_name,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        if (!empty($insertData)) {
            try {
                $chunks = array_chunk($insertData, 100);
                foreach ($chunks as $chunk) {
                    $inserted = DB::table('attendance_logs')->insertOrIgnore($chunk);
                    $insertedCount += $inserted;
                    Log::info("Migrated $inserted records to attendance_logs (offset: $offset, total inserted: $insertedCount)");
                }
            } catch (\Exception $e) {
                Log::error("Error migrating records at offset $offset: " . $e->getMessage());
            }
            $insertData = [];
        }
    }

    Log::info("Migration completed, total records inserted: $insertedCount out of $total, skipped: $skippedCount");
    return "Migrated $insertedCount records to attendance_logs successfully! Skipped $skippedCount records.";
}

    /**
     * Populate employees table from attendance data
     *
     * @return string
     */
    public function populateEmployees()
    {
//        dd('efce');
        set_time_limit(120); // Increase execution time to 120 seconds

        $employees = DB::table('_x_test_attendances')
            ->select('employee_code', 'branch_id', 'employee_name')
            ->groupBy('employee_code', 'branch_id', 'employee_name')
            ->get();

        $insertData = [];
        foreach ($employees as $emp) {
            $nameParts = explode(' ', trim($emp->employee_name ?? ''));
            $firstName = $nameParts[0] ?? null;
            $lastName = count($nameParts) > 1 ? implode(' ', array_slice($nameParts, 1)) : null;

            $insertData[] = [
                'employee_code' => $emp->employee_code,
                'branch_id' => $emp->branch_id,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'hire_date' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        if (!empty($insertData)) {
            try {
                DB::table('employees')->insertOrIgnore($insertData); // Ignore duplicates
                Log::info("Populated " . count($insertData) . " employees");
            } catch (\Exception $e) {
                Log::error("Error populating employees: " . $e->getMessage());
            }
        }

        return "Populated " . count($insertData) . " employees successfully!";
    }

    /**
     * Populate branches table from attendance logs
     *
     * @return string
     */
    public function populateBranches()
    {
        set_time_limit(120); // Increase execution time to 120 seconds

        $branches = DB::table('attendance_logs')
            ->select('branch_id', 'branch_name')
            ->groupBy('branch_id', 'branch_name')
            ->get();

        $insertData = [];
        foreach ($branches as $branch) {
            $insertData[] = [
                'branch_id' => $branch->branch_id,
                'name' => $branch->branch_name,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        DB::table('branches')->insert($insertData);
        Log::info("Populated " . count($insertData) . " branches");

        return "Populated " . count($insertData) . " branches successfully!";
    }

    /**
     * Populate performance logs based on attendance data
     *
     * @return string
     */
    public function populatePerformanceLogs()
    {
        ini_set('max_execution_time', 300);
        $batchSize = 1000; // حجم دفعة مجموعات الأيام
        $insertedCount = 0;

        // 1) عدّ إجمالي مجموعات (employee_code, branch_id, DATE(check_time))
        $totalGroupsRow = DB::selectOne("
        SELECT COUNT(*) AS cnt FROM (
            SELECT employee_code, branch_id, DATE(check_time) AS log_date
            FROM attendance_logs
            WHERE type IN ('check_in', 'check_out')
            GROUP BY employee_code, branch_id, DATE(check_time)
        ) t
    ");
        $totalGroups = (int) ($totalGroupsRow->cnt ?? 0);

        for ($offset = 0; $offset < $totalGroups; $offset += $batchSize) {
            // 2) أجمع اليوم لكل موظف/فرع مع أول دخول وآخر خروج (متوافق مع ONLY_FULL_GROUP_BY)
            $attendances = DB::select("
            SELECT
                employee_code,
                branch_id,
                DATE(check_time) AS log_date,
                MIN(CASE WHEN type = 'check_in'  THEN check_time END) AS first_check_in,
                MAX(CASE WHEN type = 'check_out' THEN check_time END) AS last_check_out
            FROM attendance_logs
            WHERE type IN ('check_in', 'check_out')
            GROUP BY employee_code, branch_id, DATE(check_time)
            ORDER BY log_date, employee_code, branch_id
            LIMIT ? OFFSET ?
        ", [$batchSize, $offset]);

            if (!$attendances) {
                continue;
            }

            $insertData = [];
            foreach ($attendances as $row) {
                $logDate       = $row->log_date;                     // YYYY-MM-DD
                $firstCheckIn  = $row->first_check_in ? Carbon::parse($row->first_check_in) : null;
                $lastCheckOut  = $row->last_check_out ? Carbon::parse($row->last_check_out) : null;

                $hours  = 0.0;
                $notes  = null;
                $status = 'absent';

                if ($firstCheckIn) {
                    $status = 'present';

                    // تحديد التأخير بناءً على زمن أول دخول فعلي
                    $lateThreshold = Carbon::parse($logDate . ' 09:30:00');
                    if ($firstCheckIn->gt($lateThreshold)) {
                        $status = 'late';
                    }

                    // حساب ساعات العمل لو في خروج
                    if ($lastCheckOut) {
                        if ($lastCheckOut->lt($firstCheckIn)) {
                            // حالة بيانات غير منطقية (خروج قبل دخول)
                            $notes = 'Check-out earlier than check-in';
                        } else {
                            $hours = $lastCheckOut->floatDiffInSeconds($firstCheckIn) / 3600.0;
                        }
                    } else {
                        $notes = 'No check-out recorded';
                    }
                } else {
                    // مفيش دخول = غياب
                    $notes = 'No check-in recorded';
                }

                $insertData[] = [
                    'employee_code' => $row->employee_code,
                    'branch_id'     => $row->branch_id,
                    // نخزّن تاريخ اليوم في الحقل (لو حقل عندك تاريخ/وقت اسمه مختلف عدّله)
                    'check_time'    => $logDate,
                    'hours_worked'  => round($hours, 2),
                    'status'        => $status,
                    'notes'         => $notes,
                    'created_at'    => now(),
                    'updated_at'    => now(),
                ];
            }

            // 3) إدراج مجزّأ لتفادي max_allowed_packet
            if (!empty($insertData)) {
                try {
                    foreach (array_chunk($insertData, 500) as $chunk) {
                        // insertOrIgnore لتفادي التكرارات لو عندك Unique Index (اقترح: unique على employee_code, branch_id, check_time)
                        $inserted = DB::table('performance_logs')->insertOrIgnore($chunk);
                        $insertedCount += $inserted;
                    }
                    Log::info("Populated $insertedCount performance logs (offset: $offset)");
                } catch (\Throwable $e) {
                    Log::error("Error populating performance logs: " . $e->getMessage());
                }
            }
        }

        Log::info("Populated performance logs for $totalGroups daily groups, total inserted: $insertedCount");
        return "Populated $insertedCount performance logs successfully!";
    }



    /**
     * Migrate old data from users and attendances to new tables
     *
     * @return string
     */
    public function migrateOldDataToNew()
    {
        set_time_limit(600); // Increase execution time to 10 minutes

        $batchSize = 500;

        $totalUsers = DB::table('users')->count();
        Log::info("Total users records to process: $totalUsers");

        $totalAttendance = DB::table('attendances')->count();
        Log::info("Total attendance records to process: $totalAttendance");

        $insertedEmployees = 0;
        $insertedAttendance = 0;
        $skippedRecords = 0;

        // Step 1: Process users table and populate employees table
        Log::info("Starting migration of users to employees table...");
        for ($offset = 0; $offset < $totalUsers; $offset += $batchSize) {
            Log::info("Processing users batch, offset: $offset, limit: $batchSize");

            $users = DB::table('users')
                ->select('id', 'name', 'type', 'created_at', 'updated_at')
                ->offset($offset)
                ->limit($batchSize)
                ->get();

            foreach ($users as $user) {
                Log::info("Processing user ID: {$user->id}, Name: {$user->name}, Type: {$user->type}");

                $nameParts = explode(' ', trim($user->name));
                $firstName = $nameParts[0] ?? '';
                $lastName = isset($nameParts[1]) ? $nameParts[1] : '';

                Log::info("Split name: first_name = $firstName, last_name = $lastName");

                $branchId = ($user->type > 8) ? 102 : 101;
                Log::info("Assigned branch_id: $branchId based on type: {$user->type}");

                $createdAt = $user->created_at ? Carbon::parse($user->created_at) : now();
                $updatedAt = $user->updated_at ? Carbon::parse($user->updated_at) : now();

                $existingEmployee = DB::table('employees')
                    ->where('first_name', $firstName)
                    ->where('last_name', $lastName)
                    ->where('branch_id', $branchId)
                    ->first();

                if ($existingEmployee) {
                    Log::info("Employee found with employee_code: {$existingEmployee->employee_code}, skipping creation");
                } else {
                    $lastCode = DB::table('employees')->max('employee_code') ?? 'HO_100000000';
                    $baseCode = (int)str_replace('HO_', '', $lastCode) + 1;
                    $newCode = 'HO_' . $baseCode;
                    Log::info("Generating new employee_code with prefix: $newCode");

                    DB::table('employees')->insert([
                        'employee_code' => $newCode,
                        'branch_id' => $branchId,
                        'first_name' => $firstName,
                        'last_name' => $lastName,
                        'created_at' => $createdAt,
                        'updated_at' => $updatedAt,
                    ]);

                    $insertedEmployees++;
                    Log::info("Successfully inserted new employee: $newCode, $firstName $lastName, branch_id: $branchId");
                }
            }
        }
        Log::info("Finished migrating users, total new employees created: $insertedEmployees");

        // Step 2: Process attendances table and populate attendance_logs
        Log::info("Starting migration of attendances to attendance_logs table...");
        for ($offset = 0; $offset < $totalAttendance; $offset += $batchSize) {
            Log::info("Processing attendances batch, offset: $offset, limit: $batchSize");

            $attendance = DB::table('attendances')
                ->select('id', 'user_id', 'check_in', 'check_out', 'created_at', 'updated_at')
                ->offset($offset)
                ->limit($batchSize)
                ->get();

            $insertData = [];
            $skippedCount = 0;
            $skippedReasons = [];

            foreach ($attendance as $record) {
                Log::info("Processing attendance record ID: {$record->id}, User ID: {$record->user_id}");

                // Handle Check in
                if ($record->check_in) {
                    Log::info("Processing Check in: {$record->check_in}");

                    $isValid = true;
                    $reason = '';

                    if (!Carbon::hasFormat($record->check_in, 'Y-m-d H:i:s')) {
                        $isValid = false;
                        $reason = 'invalid check_in format';
                        Log::warning("Invalid Check in format for record ID: {$record->id}, value: {$record->check_in}");
                    }

                    if ($isValid) {
                        $user = DB::table('users')->where('id', $record->user_id)->first();
                        if (!$user) {
                            $isValid = false;
                            $reason = 'user not found';
                            Log::warning("User not found for User ID: {$record->user_id} in record ID: {$record->id}");
                        } else {
                            Log::info("Found user: {$user->name}, Type: {$user->type}");

                            $branchId = ($user->type > 8) ? 102 : 101;
                            Log::info("Assigned branch_id: $branchId based on type: {$user->type}");

                            $nameParts = explode(' ', trim($user->name));
                            $firstName = $nameParts[0] ?? '';
                            $lastName = isset($nameParts[1]) ? $nameParts[1] : '';

                            $employee = DB::table('employees')
                                ->where('first_name', $firstName)
                                ->where('last_name', $lastName)
                                ->where('branch_id', $branchId)
                                ->first();

                            if (!$employee) {
                                $isValid = false;
                                $reason = 'employee not found';
                                Log::warning("Employee not found for user: {$user->name} in record ID: {$record->id}");
                            } else {
                                Log::info("Found employee with employee_code: {$employee->employee_code}");

                                $existingLog = DB::table('attendance_logs')
                                    ->where('employee_code', $employee->employee_code)
                                    ->where('check_time', $record->check_in)
                                    ->where('branch_id', $branchId)
                                    ->first();

                                if ($existingLog) {
                                    $isValid = false;
                                    $reason = 'duplicate record';
                                    Log::warning("Duplicate record found for employee_code: {$employee->employee_code}, check_time: {$record->check_in}, branch_id: $branchId");
                                } else {
                                    $insertData[] = [
                                        'employee_code' => $employee->employee_code,
                                        'check_time' => $record->check_in,
                                        'branch_id' => $branchId,
                                        'type' => 'check_in',
                                        'source' => 'home',
                                        'employee_name' => $user->name,
                                        'branch_name' => null,
                                        'created_at' => $record->created_at ? Carbon::parse($record->created_at) : now(),
                                        'updated_at' => $record->updated_at ? Carbon::parse($record->updated_at) : now(),
                                    ];
                                }
                            }
                        }
                    }

                    if (!$isValid) {
                        $skippedCount++;
                        $skippedReasons[$reason] = ($skippedReasons[$reason] ?? 0) + 1;
                    }
                }

                // Handle Check out
                if ($record->check_out) {
                    Log::info("Processing Check out: {$record->check_out}");

                    $isValid = true;
                    $reason = '';

                    if (!Carbon::hasFormat($record->check_out, 'Y-m-d H:i:s')) {
                        $isValid = false;
                        $reason = 'invalid check_out format';
                        Log::warning("Invalid Check out format for record ID: {$record->id}, value: {$record->check_out}");
                    }

                    if ($isValid) {
                        $user = DB::table('users')->where('id', $record->user_id)->first();
                        if (!$user) {
                            $isValid = false;
                            $reason = 'user not found';
                            Log::warning("User not found for User ID: {$record->user_id} in record ID: {$record->id}");
                        } else {
                            $branchId = ($user->type > 8) ? 102 : 101;
                            Log::info("Assigned branch_id: $branchId based on type: {$user->type}");

                            $nameParts = explode(' ', trim($user->name));
                            $firstName = $nameParts[0] ?? '';
                            $lastName = isset($nameParts[1]) ? $nameParts[1] : '';

                            $employee = DB::table('employees')
                                ->where('first_name', $firstName)
                                ->where('last_name', $lastName)
                                ->where('branch_id', $branchId)
                                ->first();

                            if (!$employee) {
                                $isValid = false;
                                $reason = 'employee not found';
                                Log::warning("Employee not found for user: {$user->name} in record ID: {$record->id}");
                            } else {
                                $existingLog = DB::table('attendance_logs')
                                    ->where('employee_code', $employee->employee_code)
                                    ->where('check_time', $record->check_out)
                                    ->where('branch_id', $branchId)
                                    ->first();

                                if ($existingLog) {
                                    $isValid = false;
                                    $reason = 'duplicate record';
                                    Log::warning("Duplicate record found for employee_code: {$employee->employee_code}, check_time: {$record->check_out}, branch_id: $branchId");
                                } else {
                                    $insertData[] = [
                                        'employee_code' => $employee->employee_code,
                                        'check_time' => $record->check_out,
                                        'branch_id' => $branchId,
                                        'type' => 'check_out',
                                        'source' => 'home',
                                        'employee_name' => $user->name,
                                        'branch_name' => null,
                                        'created_at' => $record->created_at ? Carbon::parse($record->created_at) : now(),
                                        'updated_at' => $record->updated_at ? Carbon::parse($record->updated_at) : now(),
                                    ];
                                }
                            }
                        }
                    }

                    if (!$isValid) {
                        $skippedCount++;
                        $skippedReasons[$reason] = ($skippedReasons[$reason] ?? 0) + 1;
                    }
                }
            }

            Log::info("Processed batch at offset $offset, skipped $skippedCount records, reasons: " . json_encode($skippedReasons));

            if (!empty($insertData)) {
                try {
                    $chunks = array_chunk($insertData, 100);
                    foreach ($chunks as $chunk) {
                        $inserted = DB::table('attendance_logs')->insertOrIgnore($chunk);
                        $insertedAttendance += $inserted;
                        Log::info("Successfully inserted $inserted records into attendance_logs at offset $offset");
                    }
                } catch (\Exception $e) {
                    Log::error("Error inserting records at offset $offset: " . $e->getMessage());
                }
                $insertData = [];
            }
        }

        Log::info("Migration completed: $insertedEmployees new employees created, $insertedAttendance attendance records inserted, $skippedRecords records skipped");
        return "Migration done! Created $insertedEmployees employees and inserted $insertedAttendance attendance records. Skipped $skippedRecords records.";
    }
}
