<?php
declare(strict_types=1);

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Support\Sec;

/**
 * Class MailboxController
 *
 * Handles mailbox account linking for the current user:
 * - Connect/Update (store): saves IMAP username & encrypted password.
 * - Disconnect (destroy): removes stored credentials.
 * - Status: returns masked info or JSON for UI checks.
 *
 * @package App\Http\Controllers
 */
class MailboxController extends Controller
{
    /**
     * Apply auth middleware.
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * POST /mailbox/connect
     *
     * Connect or update the user's mailbox credentials.
     * Stores IMAP username as-is and password encrypted at rest.
     *
     * @param  Request  $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'imap_username' => ['required','email'],
            'imap_password' => ['nullable','string','min:6'], // allow empty to keep existing
        ]);

        $u = $request->user();
        $u->imap_username = $data['imap_username'];

        // Only update password when provided (to allow "keep existing" flow)
        if (!empty($data['imap_password'])) {
            $u->imap_password_encrypted = Sec::enc($data['imap_password']);
        }

        $u->save();

        return $request->wantsJson()
            ? response()->json(['ok' => true])
            : back()->with('ok', 'Mailbox connected/updated securely.');
    }

    /**
     * DELETE /mailbox
     * (or POST /mailbox/disconnect if you prefer)
     *
     * Remove stored IMAP credentials for the current user.
     *
     * @param  Request  $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
    public function destroy(Request $request)
    {
        $u = $request->user();
        $u->imap_username = null;
        $u->imap_password_encrypted = null;
        $u->save();

        return $request->wantsJson()
            ? response()->json(['ok' => true])
            : back()->with('ok', 'Mailbox disconnected.');
    }

    /**
     * GET /mailbox/status
     *
     * Returns masked mailbox status for the current user.
     * Useful for AJAX checks on settings pages.
     *
     * @param  Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function status(Request $request)
    {
        $u = $request->user();
        $connected = filled($u->imap_username) && filled($u->imap_password_encrypted);

        $masked = null;
        if ($u->imap_username) {
            [$local, $domain] = array_pad(explode('@', $u->imap_username, 2), 2, '');
            $masked = (strlen($local) ? substr($local, 0, 1).'****' : '****') . '@' . $domain;
        }

        return response()->json([
            'connected' => $connected,
            'username'  => $masked,
        ]);
    }
}
