<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class MyAttendanceController extends Controller
{
    /**
     * صفحة الموظف (FullCalendar + DataTable)
     */
    public function page(Request $request)
    {
        return view('pages.home_attandance.me');
    }

    /**
     * Events لِـ FullCalendar
     * يستقبل ?start=YYYY-MM-DD&end=YYYY-MM-DD (FullCalendar بيبعتهم تلقائياً)
     */
    public function events(Request $request)
    {

        $employee = $request->user()->employee;
        if (!($employee && $employee->employee_code)) {
            // رجّع 200 مع مصفوفة فاضية علشان الـ Calendar مايفشلش
            return response()->json([]);
        }
        abort_unless($employee && $employee->employee_code, 403, 'لا توجد بطاقة موظف مرتبطة بالحساب.');

        $start = Carbon::parse($request->query('start', now()->startOfMonth()))->startOfDay();
        $end   = Carbon::parse($request->query('end',   now()->endOfMonth()))->endOfDay();

        $rows = DB::table('attendance_daily')
            ->where('employee_code', $employee->employee_code)
            ->whereBetween('work_date', [$start->toDateString(), $end->toDateString()])
            ->orderBy('work_date')
            ->get();

        $events = [];
        foreach ($rows as $row) {
            if ($row->first_in) {
                $events[] = [
                    'title' => 'Check In',
                    'start' => $row->first_in, // ISO datetime
                    'color' => '#2ecc71',
                ];
            }
            if ($row->last_out) {
                $events[] = [
                    'title' => 'Check Out',
                    'start' => $row->last_out,
                    'color' => '#e74c3c',
                ];
            }
        }

        return response()->json($events);
    }

    /**
     * DataTables (Server-side) لسجلات حضور الموظف اليومية
     * فلاتر اختيارية: status/date_from/date_to/branch_id
     */
    public function dt(Request $request)
    {
        // ===== Inputs (بنفس ستايلك)
        $searchRaw = $request->input('search', '');
        $search = is_array($searchRaw)
            ? trim((string)($searchRaw['value'] ?? ''))
            : trim((string)$searchRaw);

        $status   = (string) $request->input('status', '');       // present|late|absent|in_progress
        $dateFrom = (string) $request->input('date_from', '');
        $dateTo   = (string) $request->input('date_to', '');
        $branchId = (string) $request->input('branch_id', '');    // اختياري

        // ترتيب الأعمدة: [0 work_date][1 first_in][2 last_out][3 hours][4 sources][5 status][6 actions]
        $orderColIdx = (int) $request->input('order.0.column', 0);
        $orderDir    = strtolower((string) $request->input('order.0.dir', 'desc')) === 'asc' ? 'asc' : 'desc';
        $map = [
            0 => 'work_date',
            1 => 'first_in',
            2 => 'last_out',
            3 => 'total_minutes',
            4 => 'sources',
            5 => 'status_value',
            6 => null, // actions
        ];
        $orderBy = $map[$orderColIdx] ?? 'work_date';

        // Pagination
        $start  = max(0, (int) $request->input('start', 0));
        $length = (int) $request->input('length', 10);
        if ($length <= 0 || $length > 1000) $length = 10;

        $employee = $request->user()->employee;
        abort_unless($employee && $employee->employee_code, 403, 'لا توجد بطاقة موظف مرتبطة بالحساب.');

        // ===== Base Query (attendance_daily) + حقول مشتقة
        $base = DB::table('attendance_daily as ad')
            ->selectRaw("
                ad.employee_code,
                ad.branch_id,
                ad.work_date,
                ad.first_in,
                ad.last_out,
                CONCAT(COALESCE(ad.first_in_source,'-'),' / ',COALESCE(ad.last_out_source,'-')) as sources,
                CASE WHEN ad.first_in IS NOT NULL AND ad.last_out IS NOT NULL
                     THEN TIMESTAMPDIFF(MINUTE, ad.first_in, ad.last_out)
                     ELSE 0 END as total_minutes,
                CASE
                  WHEN ad.first_in IS NULL THEN 'absent'
                  WHEN ad.first_in IS NOT NULL AND ad.last_out IS NULL THEN 'in_progress'
                  WHEN TIME(ad.first_in) > '09:00:00' THEN 'late'
                  ELSE 'present'
                END as status_value
            ")
            ->where('ad.employee_code', $employee->employee_code);

        // فلاتر التاريخ/الفرع/الحالة
        $base->when($dateFrom !== '' && $dateTo !== '', function ($q) use ($dateFrom, $dateTo) {
            $from = Carbon::parse($dateFrom)->toDateString();
            $to   = Carbon::parse($dateTo)->toDateString();
            $q->whereBetween('ad.work_date', [$from, $to]);
        })
            ->when($branchId !== '', fn($q) => $q->where('ad.branch_id', $branchId))
            ->when($status !== '', function ($q) use ($status) {
                // status_value حقل مشتق، فنستخدم having
                $q->having('status_value', '=', $status);
            });

        // إجمالي قبل البحث العام
        $recordsTotal = (clone $base)->count();

        // البحث العام
        $filtered = (clone $base);
        if ($search !== '') {
            $filtered->where(function ($w) use ($search) {
                $w->where('ad.work_date', 'like', "%{$search}%")
                    ->orWhereRaw("CONCAT(COALESCE(ad.first_in_source,'-'),' / ',COALESCE(ad.last_out_source,'-')) like ?", ["%{$search}%"]);
            });
        }

        // إجمالي بعد البحث
        $recordsFiltered = (clone $filtered)->count();

        // ترتيب
        if ($orderBy === 'sources' || $orderBy === 'status_value') {
            $filtered->orderByRaw("$orderBy $orderDir");
        } else {
            $filtered->orderBy($orderBy, $orderDir);
        }

        // Fetch
        $rows = $filtered->skip($start)->take($length)->get();

        // Transform
        $data = $rows->map(function ($r) {
            $hoursTxt = '-';
            if ((int)$r->total_minutes > 0) {
                $h = intdiv((int)$r->total_minutes, 60);
                $m = ((int)$r->total_minutes) % 60;
                $hoursTxt = sprintf('%dh %dm', $h, $m);
            }

            $badge = match ($r->status_value) {
                'present'     => '<span class="badge bg-success">Present</span>',
                'late'        => '<span class="badge bg-warning">Late</span>',
                'in_progress' => '<span class="badge bg-info">In&nbsp;Progress</span>',
                default       => '<span class="badge bg-danger">Absent</span>',
            };

            return [
                'work_date'     => $r->work_date,
                'first_in'      => $r->first_in ? Carbon::parse($r->first_in)->format('H:i') : '-',
                'last_out'      => $r->last_out ? Carbon::parse($r->last_out)->format('H:i') : '-',
                'total_minutes' => (int) $r->total_minutes,
                'hours'         => $hoursTxt,
                'sources'       => $r->sources,
                'status'        => $badge,
                'status_value'  => (string) $r->status_value,
            ];
        });

        return response()->json([
            'draw'            => (int) $request->input('draw'),
            'recordsTotal'    => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data'            => $data,
        ]);
    }
}
