<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class NotificationCenterController extends Controller
{
    public function index(Request $request)
    {
        $app = rtrim(config('app.url'), '/');

        $items = $request->user()->notifications()
            ->latest()->limit(50)->get()
            ->map(function ($n) use ($app) {
                $d   = (array) $n->data;
                $rel = (string) ($d['open_url'] ?? '/');     // path نسبي محفوظ داخل الإشعار
                $abs = $app . '/' . ltrim($rel, '/');        // مطلق مبني على APP_URL

                return [
                    'id'         => $n->id,
                    'title'      => $d['title'] ?? 'تنبيه',
                    'body'       => $d['body'] ?? '',
                    'open_url'   => $rel,                     // نسبي
                    'abs_url'    => $abs,                     // مطلق (JSON/UI)
                    'status'     => $d['status'] ?? null,
                    'created_at' => optional($n->created_at)->toIso8601String(),
                    'read_at'    => optional($n->read_at)->toIso8601String(),
                ];
            });

        if ($request->expectsJson() || $request->boolean('json')) {
            return response()->json(['data' => $items]);
        }

        return view('pages.notifications.index', ['items' => $items]);
    }

    public function unreadCount(Request $request)
    {
        return response()->json([
            'count' => $request->user()->unreadNotifications()->count()
        ]);
    }

    public function markRead(Request $request, string $id)
    {
        $n = $request->user()->notifications()->where('id', $id)->firstOrFail();
        if (!$n->read_at) $n->markAsRead();
        return response()->json(['ok' => true]);
    }

    public function markAllRead(Request $request)
    {
        $request->user()->unreadNotifications->markAsRead();
        return response()->json(['ok' => true]);
    }

    /** فتح بالإيدى: يعَلِّم كمقروء ويحوّل على APP_URL */
    public function openId(Request $request, string $id)
    {
        $n = $request->user()->notifications()->where('id', $id)->firstOrFail();
        if (!$n->read_at) $n->markAsRead();

        $d    = (array) $n->data;
        $rel  = (string) ($d['open_url'] ?? '/');       // مسار نسبي
        $href = $this->absFromAppUrl($rel);             // مطلق من APP_URL

        return redirect()->away($href);
    }

    /** فتح بـ ?m=model:id (للـ live لما مفيش notification id) */
    public function open(Request $r)
    {
        $m = (string) $r->query('m', '');
        if (!$m || !str_contains($m, ':')) {
            return redirect()->route('dashboard');
        }

        [$model, $idStr] = explode(':', $m, 2);
        $model = strtolower(trim($model));
        $id    = (int) trim($idStr);

        if ($id <= 0) return redirect()->route('dashboard');

        // علّم أي إشعارات مطابقة كمقروء
        $r->user()->unreadNotifications()
            ->where('data->model', $model)
            ->where('data->model_id', $id)
            ->get()
            ->each->markAsRead();

        // هات مسار نسبي حسب الموديل
        $rel = match ($model) {
            'vacation'          => route('vacations.show', ['vacation' => $id], false),
            'attendance_permit' => route('permits.show.page', ['permit' => $id], false),
            default             => '/',
        };

        return redirect()->away($this->absFromAppUrl($rel));
    }

    /* ================== Helpers ================== */

    /** يبني URL مطلق من APP_URL لمسار نسبي أو يُعيد المطلق كما هو */
    private function absFromAppUrl(string $pathOrUrl): string
    {
        if (preg_match('/^https?:\/\//i', $pathOrUrl)) {
            return $pathOrUrl; // بالفعل مطلق
        }
        $base = rtrim(config('app.url'), '/');
        $path = '/' . ltrim($pathOrUrl ?: '/', '/');
        return $base . $path;
    }
}
