<?php

namespace App\Http\Controllers;

use App\Models\AttendancePermit;
use App\Models\AttendancePermitType;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Gate;
use App\Services\AttendancePermitService;
// ...existing code...
use App\Events\PermitRequested;
use App\Events\PermitStatusChanged;

class PermitController extends Controller
{
    public function __construct(private AttendancePermitService $service) {}

    /* ========== Views ========== */
    public function index(Request $r)
    {
        $this->authorize('viewAny', AttendancePermit::class);
        $types = AttendancePermitType::orderBy('name')->get(['id', 'name', 'display_name']);
        return view('pages.permits.index', compact('types'));
    }

    public function showPage($id)
    {
        $permit = AttendancePermit::with(['employee.user', 'type', 'leaderApprovedBy', 'hrApprovedBy'])
            ->findOrFail($id);
        $this->authorize('view', $permit);

        return view('pages.permits.show', [
            'permit'             => $permit,
            'can_leader_approve' => auth()->user()->can('leaderApprove', $permit),
            'can_hr_approve'     => auth()->user()->can('hrApprove', $permit),
            'can_update'         => auth()->user()->can('update', $permit),
            'can_cancel'         => auth()->user()->can('cancel', $permit),
        ]);
    }

    public function editPage($id)
    {
        $permit = AttendancePermit::with(['employee.user', 'type'])->findOrFail($id);
        $this->authorize('update', $permit);
        $types = AttendancePermitType::orderBy('display_name')->get();
        return view('pages.permits.edit', compact('permit', 'types'));
    }

    /* ========== DataTables ========== */
    public function dt(Request $request)
    {
        $search = trim($request->input('search.value', ''));
        $typeId = $request->input('type', '');
        $status = $request->input('status', '');
        $dateFrom = $request->input('date_from', '');
        $dateTo = $request->input('date_to', '');

        $orderColIdx = (int) $request->input('order.0.column', 0);
        $orderDir = $request->input('order.0.dir', 'desc') === 'asc' ? 'asc' : 'desc';

        $map = [
            0 => 'attendance_permits.id',
            1 => 'employee_name',
            2 => 'attendance_permit_types.name',
            3 => 'attendance_permits.start_datetime',
            4 => 'attendance_permits.end_datetime',
            5 => 'attendance_permits.total_minutes',
            6 => 'attendance_permits.status',
        ];
        $orderBy = $map[$orderColIdx] ?? 'attendance_permits.id';

        $user = $request->user();
        $employee = $user->employee;
        $base = AttendancePermit::query()->with(['employee.user', 'type']);

        // == Role filtering ==
        if ($user->hasAnyRole(['Admin', 'Manager', 'HR'])) {
            // No restrictions
        } elseif ($user->hasRole('Team Lead')) {
            abort_if(!$employee?->id, 403, 'لا توجد بطاقة موظف مرتبطة.');
            $base->where(function ($q) use ($employee) {
                $q->where('employee_id', $employee->id)
                    ->orWhereHas(
                        'employee',
                        fn($w) =>
                        $w->where('department_id', $employee->department_id)
                    );
            });
        } else {
            abort_if(!$employee?->id, 403, 'لا توجد بطاقة موظف مرتبطة.');
            $base->where('employee_id', $employee->id);
        }

        // == Filters ==
        if ($typeId) $base->where('permit_type_id', $typeId);
        if ($status) $base->whereRaw('LOWER(status) = ?', [strtolower($status)]);
        if ($dateFrom && $dateTo) {
            $from = Carbon::parse($dateFrom)->startOfDay();
            $to = Carbon::parse($dateTo)->endOfDay();
            $base->whereBetween('start_datetime', [$from, $to]);
        }

        $recordsTotal = (clone $base)->count();

        if ($search !== '') {
            $base->where(function ($w) use ($search) {
                $w->whereHas(
                    'employee.user',
                    fn($q) =>
                    $q->where('name', 'like', "%$search%")
                )->orWhereHas(
                    'type',
                    fn($q) =>
                    $q->where('name', 'like', "%$search%")->orWhere('display_name', 'like', "%$search%")
                )->orWhere('member_reason', 'like', "%$search%");
            });
        }

        $recordsFiltered = (clone $base)->count();

        if ($orderBy === 'employee_name') {
            $base->leftJoin('employees', 'employees.id', '=', 'attendance_permits.employee_id')
                ->orderBy('employees.first_name', $orderDir)
                ->orderBy('employees.last_name', $orderDir)
                ->select('attendance_permits.*');
        } elseif ($orderBy === 'attendance_permit_types.name') {
            $base->leftJoin('attendance_permit_types', 'attendance_permit_types.id', '=', 'attendance_permits.permit_type_id')
                ->orderBy('attendance_permit_types.name', $orderDir)
                ->select('attendance_permits.*');
        } else {
            $base->orderBy($orderBy, $orderDir);
        }

        $rows = $base->skip($request->input('start', 0))
            ->take($request->input('length', 10))
            ->get();

        $authUser = auth()->user();
        $now = now();

        $data = $rows->map(function (AttendancePermit $p) use ($authUser, $now) {
            $status = strtolower($p->status);
            $isFinal = in_array($status, ['hr_approved', 'leader_rejected', 'hr_rejected', 'cancelled_by_member'])
                || ($status === 'leader_approved' && !$p->needs_hr);
            return [
                'id' => $p->id,
                'employee_name' => $p->employee->user->name ?? '—',
                'type_name' => $p->type->display_name ?? $p->type->name ?? '—',
                'start_datetime' => optional($p->start_datetime)->format('Y-m-d H:i'),
                'end_datetime' => optional($p->end_datetime)->format('Y-m-d H:i'),
                'total_minutes' => $p->total_minutes,
                'status' => $status,
                'needs_hr' => $p->needs_hr,
                'can_update' => $authUser->can('update', $p),
                'can_cancel' => $authUser->can('cancel', $p),
                'can_leader_approve' => $authUser->can('leaderApprove', $p),
                'can_hr_approve' => $authUser->can('hrApprove', $p),
                'can_reject' => $authUser->can('reject', $p),
                'is_final' => $isFinal,
                'hr_window_open' => $now->lt(optional($p->start_datetime)->copy()->startOfMinute()),
            ];
        });

        return response()->json([
            'draw' => (int) $request->input('draw'),
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
        ]);
    }

    /* ========== CRUD ========== */
    public function store(Request $r)
    {
        $this->authorize('create', AttendancePermit::class);
        $employee = $r->user()->employee;
        abort_if(!$employee?->id, 422, 'لا توجد بطاقة موظف.');

        $data = $r->validate([
            'permit_type_id' => ['required', 'exists:attendance_permit_types,id'],
            'start_datetime' => ['required', 'date'],
            'end_datetime'   => ['required', 'date', 'after:start_datetime'],
            'member_reason'  => ['nullable', 'string', 'max:1000'],
        ]);

        $permit = $this->service->create($data, $employee->id, $r->user()->id)->load(['employee.user', 'type']);

        // broadcast + notification (mirror vacations)
        $recipientIds = $this->resolveRecipientsForNewRequest($permit);
        event(new PermitRequested($permit, $recipientIds));

        $this->notifyUsers($recipientIds, new \App\Notifications\AttendancePermitStatusNotification(
            permit: $permit,
            action: 'submitted',
            actor: $r->user()
        ));

        $msg = 'تم الإنشاء';
        return $r->expectsJson()
            ? response()->json(['ok' => true, 'id' => $permit->id, 'message' => $msg])
            : redirect()->route('permits.show.page', $permit)->with('success', $msg);
    }

    public function update(Request $r, AttendancePermit $permit)
    {
        $this->authorize('update', $permit);

        $data = $r->validate([
            'permit_type_id' => ['required', 'exists:attendance_permit_types,id'],
            'start_datetime' => ['required', 'date'],
            'end_datetime'   => ['required', 'date', 'after:start_datetime'],
            'member_reason'  => ['nullable', 'string', 'max:1000'],
        ]);

        $this->service->update($permit, $data);
        return $r->expectsJson()
            ? response()->json(['ok' => true, 'id' => $permit->id, 'message' => 'تم التحديث'])
            : redirect()->route('permits.show.page', $permit)->with('success', 'تم التحديث');
    }

    public function destroy(Request $r, AttendancePermit $permit)
    {
        $this->authorize('cancel', $permit);
        $this->service->cancelByMember($permit);

        $permit->refresh()->load(['employee.user', 'type']);

        $recipientIds = $this->resolveRecipientsForStatusChange($permit, actorIsHr: false, actorIsLeader: false);
        event(new PermitStatusChanged(
            permit: $permit,
            status: 'cancelled_by_member',
            actorName: $r->user()->name,
            recipientIds: $recipientIds
        ));

        $this->notifyUsers($recipientIds, new \App\Notifications\AttendancePermitStatusNotification(
            permit: $permit,
            action: 'cancelled_by_member',
            actor: $r->user()
        ));

        return $r->expectsJson()
            ? response()->json(['ok' => true, 'message' => 'تم الإلغاء'])
            : redirect()->route('permits.index')->with('success', 'تم الإلغاء');
    }

    /* ========== Approvals ========== */
    public function leaderApprove(Request $r, AttendancePermit $permit)
    {
        $this->authorize('leaderApprove', $permit);

        $data = $r->validate([
            'decision'      => ['required', Rule::in(['approve', 'reject'])],
            'leader_reason' => ['nullable', 'string', 'max:1000'],
        ]);

        if ($data['decision'] === 'approve') {
            $this->service->leaderApprove($permit, $data['leader_reason'] ?? null, $r->user()->id);
            $msg = 'تم اعتماد التيم ليدر.';
        } else {
            $this->service->reject($permit, $data['leader_reason'] ?? null, 'leader', $r->user()->id);
            $msg = 'تم رفض الطلب (TL).';
        }

        $permit->refresh()->load(['employee.user', 'type']);

        $recipientIds = $this->resolveRecipientsForStatusChange($permit, actorIsHr: false, actorIsLeader: true);
        event(new PermitStatusChanged(
            permit: $permit,
            status: 'approved',
            actorName: $r->user()->name,
            recipientIds: $recipientIds
        ));

        $this->notifyUsers($recipientIds, new \App\Notifications\AttendancePermitStatusNotification(
            permit: $permit,
            action: $data['decision'] === 'approve' ? 'leader_approved' : 'leader_rejected',
            actor: $r->user()
        ));

        return $r->expectsJson()
            ? response()->json(['ok' => true, 'status' => $permit->fresh()->status, 'message' => $msg])
            : back()->with('success', $msg);
    }


    public function hrApprove(Request $r, AttendancePermit $permit)
    {
        $this->authorize('hrApprove', $permit);
        $data = $r->validate([
            'decision'  => ['required', Rule::in(['approve', 'reject'])],
            'hr_reason' => ['nullable', 'string', 'max:1000'],
        ]);

        $decision = $data['decision'] === 'approve'
            ? $this->service->hrApprove($permit, $data['hr_reason'] ?? null, $r->user()->id)
            : $this->service->reject($permit, $data['hr_reason'] ?? null, 'hr', $r->user()->id);

        $permit->refresh()->load(['employee.user', 'type']);

        $isHr = $data['decision'] === 'approve';
        $recipientIds = $this->resolveRecipientsForStatusChange($permit, actorIsHr: $isHr, actorIsLeader: !$isHr);

        event(new PermitStatusChanged(
            permit: $permit,
            status: 'approved',
            actorName: $r->user()->name,
            recipientIds: $recipientIds
        ));

        $this->notifyUsers($recipientIds, new \App\Notifications\AttendancePermitStatusNotification(
            permit: $permit,
            action: $data['decision'] === 'approve' ? 'hr_approved' : 'hr_rejected',
            actor: $r->user()
        ));

        return $r->expectsJson()
            ? response()->json(['ok' => true, 'status' => $permit->fresh()->status, 'message' => 'تم التنفيذ'])
            : back()->with('success', 'تم التنفيذ');
    }

    /* ===== Helpers: recipients + notify (copied/adjusted from VacationController) ===== */

    private function notifyUsers(array $recipientIds, \Illuminate\Notifications\Notification $notification): void
    {
        if (empty($recipientIds)) return;
        try {
            $users = User::whereIn('id', $recipientIds)->get();
            foreach ($users as $u) {
                try {
                    $u->notify($notification);
                } catch (\Throwable $e) {
                    report($e);
                }
            }
        } catch (\Throwable $e) {
            // ignore
        }
    }

    private function resolveRecipientsForNewRequest(AttendancePermit $permit): array
    {
        $empUserId = (int) ($permit->employee?->user_id ?? 0);
        $deptId = (int) ($permit->employee?->department_id ?? 0);

        $leaderIds = $this->findTeamLeadersByDepartment($deptId);
        $hrIds = $this->findHrAndAdmins();

        return $this->uniqueIds(array_merge([$empUserId], $leaderIds, $hrIds));
    }

    private function resolveRecipientsForStatusChange(AttendancePermit $permit, bool $actorIsHr, bool $actorIsLeader): array
    {
        $empUserId = (int) ($permit->employee?->user_id ?? 0);
        $deptId = (int) ($permit->employee?->department_id ?? 0);

        $base = [$empUserId];

        if ($actorIsLeader) {
            $base = array_merge($base, $this->findHrAndAdmins());
        } elseif ($actorIsHr) {
            $base = array_merge($base, $this->findTeamLeadersByDepartment($deptId));
        }

        return $this->uniqueIds($base);
    }

    private function findTeamLeadersByDepartment(int $departmentId): array
    {
        if ($departmentId <= 0) return [];

        $ids = [];
        try {
            $ids = array_merge($ids, User::role('Team Lead')
                ->whereHas('leaderOfDepartments', fn($q) => $q->where('departments.id', $departmentId))
                ->pluck('id')->all());
        } catch (\Throwable $e) {
        }

        try {
            $ids = array_merge($ids, User::role('Team Lead')
                ->whereHas('employee', fn($q) => $q->where('department_id', $departmentId))
                ->pluck('id')->all());
        } catch (\Throwable $e) {
        }

        return $this->uniqueIds($ids);
    }

    private function findHrAndAdmins(): array
    {
        try {
            return User::role(['HR', 'Admin'])->pluck('id')->all();
        } catch (\Throwable $e) {
            return [];
        }
    }

    private function uniqueIds(array $ids): array
    {
        $ids = array_map('intval', array_filter($ids, fn($v) => (int)$v > 0));
        sort($ids);
        return array_values(array_unique($ids));
    }
}
