<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProfileUpdateRequest;
use App\Http\Requests\AvatarUpdateRequest;
use App\Services\ProfileTabService;
use App\Services\AttendanceService;
use App\Services\PerformanceService;
use App\Services\AvatarService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\View;

class ProfileController extends Controller
{
    protected $profileTabService;
    protected $attendanceService;
    protected $performanceService;
    protected $avatarService;

    public function __construct(
        ProfileTabService $profileTabService,
        AttendanceService $attendanceService,
        PerformanceService $performanceService,
        AvatarService $avatarService
    ) {
        $this->profileTabService = $profileTabService;
        $this->attendanceService = $attendanceService;
        $this->performanceService = $performanceService;
        $this->avatarService = $avatarService;
    }
    /**
     * Display the user's profile page with dynamic navigation.
     */
    public function index()
    {
        $user = Auth::user();
        $navigationTabs = $this->profileTabService->getNavigationTabs($user);

        // Get attendance data for the authenticated user
        $attendanceStats = $this->attendanceService->getAttendanceStats($user);
        $todayAttendance = $this->attendanceService->getTodayAttendance($user);
        $recentAttendanceRecords = $this->attendanceService->getRecentAttendanceRecords($user, 5);
        $teamAttendance = $this->attendanceService->getTeamAttendance($user);

        // Get performance data for default week and month views
        $weekPerformance = $this->performanceService->getInitialPerformanceData($user, 'week');
        $monthPerformance = $this->performanceService->getInitialPerformanceData($user, 'month');

        return view('pages.profile', compact(
            'navigationTabs',
            'attendanceStats',
            'todayAttendance',
            'recentAttendanceRecords',
            'teamAttendance',
            'weekPerformance',
            'monthPerformance'
        ));
    }
    /**
     * Display the user's profile form.
     */
    public function edit(Request $request): View
    {
        return view('profile.edit', [
            'user' => $request->user(),
        ]);
    }

    /**
     * Update the user's profile information.
     */
    public function update(ProfileUpdateRequest $request): RedirectResponse
    {
        $request->user()->fill($request->validated());

        if ($request->user()->isDirty('email')) {
            $request->user()->email_verified_at = null;
        }

        $request->user()->save();

        return Redirect::route('profile.edit')->with('status', 'profile-updated');
    }

    /**
     * Delete the user's account.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $request->validateWithBag('userDeletion', [
            'password' => ['required', 'current_password'],
        ]);

        $user = $request->user();

        Auth::logout();

        $user->delete();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return Redirect::to('/');
    }


    /**
     * Get user performance data.
     */
    public function getPerformanceData(Request $request)
    {
        $user = Auth::user();
        $type = $request->get('type', 'week');
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');

        $performanceData = $this->performanceService->getPerformanceData($user, $type, $startDate, $endDate);

        return response()->json($performanceData);
    }


    /**
     * AJAX endpoint to get tab data dynamically.
     */
    public function getTabData(Request $request, $tab)
    {
        $user = Auth::user();
        $data = $this->profileTabService->getTabData($user, $tab);

        if ($data === null) {
            return response()->json(['error' => 'Tab not found or access denied'], 404);
        }

        return response()->json($data);
    }

    public function updateAvatar(AvatarUpdateRequest $request)
    {
        $this->avatarService->updateUserAvatar(Auth::user(), $request->file('avatar'));

        return response()->json([
            'success' => true,
            'message' => 'Avatar updated successfully!'
        ]);
    }

    public function allAttendance(Request $request)
    {
        $user = Auth::user();
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');

        $allAttendanceRecords = $this->attendanceService->getAllAttendanceRecords($user, 15, $startDate, $endDate);

        return view('pages.profile.all-attendance', compact('allAttendanceRecords'));
    }

}
