<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use App\Models\Department;
use Spatie\Permission\Models\Role;

class RoleController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth']);
    }

    /** صفحة الجدول */
    public function index(Request $request)
    {
        $roles       = Role::orderBy('name')->get(['id','name','guard_name']);
        $departments = Department::orderBy('name')->get(['id','name','parent_id','depth']);
        // NOTE: مسار الصفحة حسب مشروعك
        return view('pages.roles.index', compact('roles','departments'));
    }

    /** DataTables (النسخة المعتمدة) */
    public function dt(Request $request)
    {
        // ====== Inputs (تطبيع آمن) ======
        // search ممكن تيجي من فلاتر الواجهة كسلسلة، وممكن تيجي من DT كأوبجكت {value:..}
        $searchRaw = $request->input('search', '');
        $search = is_array($searchRaw)
            ? trim((string)($searchRaw['value'] ?? ''))
            : trim((string)$searchRaw);

        // roles, departments دايمًا مصفوفات
        $roles       = array_values(array_filter((array) $request->input('roles', [])));
        $has         = (string) $request->input('has', 'all'); // all|with|without
        $departments = array_values(array_filter((array) $request->input('departments', [])));

        // ترتيب
        $orderColIdx = (int) $request->input('order.0.column', 2);
        $orderDir    = strtolower((string) $request->input('order.0.dir', 'asc')) === 'desc' ? 'desc' : 'asc';
        // الأعمدة: [0 checkbox][1 id][2 name][3 email][4 roles][5 actions]
        $orderMap = [
            1 => 'users.id',
            2 => 'users.name',
            3 => 'users.email',
            4 => null,
            5 => null,
        ];
        $orderBy = $orderMap[$orderColIdx] ?? 'users.name';

        // Pagination
        $start  = max(0, (int) $request->input('start', 0));
        $length = (int) $request->input('length', 10);
        if ($length <= 0 || $length > 1000) $length = 10;

        // ====== Base Query ======
        $base = User::query()
            ->with([
                'roles:id,name',
                'employee:id,user_id,department_id',
                'employee.department:id,name'
            ]);

        // Team Lead restriction: only view members of their managed departments
        $currentUser = $request->user();
        if ($currentUser->hasRole('Team Lead') && !$currentUser->hasRole(['Admin', 'HR'])) {
            $managedDepartments = $currentUser->getManagedDepartments()->pluck('id')->toArray();
            if (!empty($managedDepartments)) {
                $base->whereHas('employee', function($q) use ($managedDepartments) {
                    $q->whereIn('department_id', $managedDepartments);
                });
            } else {
                // If team lead has no managed departments, show no users
                $base->whereRaw('1 = 0');
            }
        }

        // has roles
        if ($has === 'with') {
            $base->whereHas('roles');
        } elseif ($has === 'without') {
            $base->whereDoesntHave('roles');
        }

        // filter by roles
        if (!empty($roles)) {
            $base->whereHas('roles', function($q) use ($roles){
                $q->whereIn('name', $roles);
            });
        }

        // filter by departments (via employee)
        if (!empty($departments)) {
            $base->whereHas('employee', function($q) use ($departments){
                $q->whereIn('department_id', $departments);
            });
        }

        // إجمالي قبل البحث العام
        $recordsTotal = (clone $base)->count();

        // تطبيق البحث العام
        $filtered = (clone $base);
        if ($search !== '') {
            $filtered->where(function($w) use ($search){
                $w->where('name','like',"%{$search}%")
                  ->orWhere('email','like',"%{$search}%");
            });
        }

        $recordsFiltered = (clone $filtered)->count();

        // ترتيب
        if ($orderBy) {
            $filtered->orderBy($orderBy, $orderDir);
        } else {
            $filtered->orderBy('users.name', 'asc');
        }

        // Pagination + Fetch
        $rows = $filtered->skip($start)->take($length)->get(['id','name','email']);

        // Transform
        $data = $rows->map(function(User $u){
            return [
                'id'    => $u->id,
                'name'  => $u->name,
                'email' => $u->email,
                'roles' => $u->roles->pluck('name')->values(), // array of strings
            ];
        });

        return response()->json([
            'draw'            => (int) $request->input('draw'),
            'recordsTotal'    => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data'            => $data,
        ]);
    }

    /** نسخة قديمة – لو لسه بتستخدم route('roles.datatable') */
    public function datatable(Request $request)
    {
        $draw   = (int) $request->input('draw', 1);
        $start  = max(0, (int) $request->input('start', 0));
        $length = (int) $request->input('length', 10);
        if ($length <= 0 || $length > 1000) $length = 10;

        // search قد تيجي كobject من DT
        $searchRaw = $request->input('search', '');
        $search = is_array($searchRaw) ? trim((string)($searchRaw['value'] ?? '')) : trim((string)$searchRaw);

        $roles    = array_values(array_filter((array) $request->input('roles', [])));
        $has      = (string) $request->input('has', 'all');
        $deptIds  = array_values(array_filter((array) $request->input('departments', [])));

        $orderColIdx = (int) data_get($request->input('order'), '0.column', 2);
        $orderDir    = strtolower((string) data_get($request->input('order'), '0.dir', 'asc')) === 'desc' ? 'desc' : 'asc';
        $columns     = ['_check','id','name','email','roles','_actions'];
        $orderBy     = $columns[$orderColIdx] ?? 'name';
        if (!in_array($orderBy, ['id','name','email'], true)) $orderBy = 'name';

        $q = User::query()
            ->with('roles:id,name');

        // Team Lead restriction: only view members of their managed departments
        $currentUser = $request->user();
        if ($currentUser->hasRole('Team Lead') && !$currentUser->hasRole(['Admin', 'HR'])) {
            $managedDepartments = $currentUser->getManagedDepartments()->pluck('id')->toArray();
            if (!empty($managedDepartments)) {
                $q->whereHas('employee', function($query) use ($managedDepartments) {
                    $query->whereIn('department_id', $managedDepartments);
                });
            } else {
                $q->whereRaw('1 = 0');
            }
        }

        if ($search !== '') {
            $q->where(function($x) use ($search){
                $x->where('name','like',"%{$search}%")
                  ->orWhere('email','like',"%{$search}%");
            });
        }

        if ($has === 'with')   $q->has('roles');
        if ($has === 'without')$q->doesntHave('roles');

        if (!empty($roles)) {
            $q->whereHas('roles', fn($h)=> $h->whereIn('name',$roles));
        }

        if (!empty($deptIds)) {
            $q->whereHas('employee', fn($h)=> $h->whereIn('department_id', $deptIds));
        }

        $recordsTotal    = User::count();        // قبل أي فلاتر (لو عايز نفس سلوك DT التقليدي)
        $recordsFiltered = (clone $q)->count();

        $rows = $q->orderBy($orderBy, $orderDir)
            ->skip($start)->take($length)
            ->get(['id','name','email']);

        $data = $rows->map(fn(User $u)=>[
            'id'    => $u->id,
            'name'  => $u->name,
            'email' => $u->email,
            'roles' => $u->roles->pluck('name')->values(),
        ]);

        return response()->json([
            'draw'            => $draw,
            'recordsTotal'    => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data'            => $data,
        ]);
    }

    /** Attach role */
    public function attach(Request $request)
    {
        $data = $request->validate([
            'user_id' => ['required','exists:users,id'],
            'role'    => ['required','exists:roles,name'],
        ]);

        $currentUser = $request->user();

        // Team leads can only manage Employee role for users in their departments
        if ($currentUser->hasRole('Team Lead') && !$currentUser->hasRole(['Admin', 'HR'])) {
            if ($data['role'] !== 'Employee') {
                return response()->json(['success' => false, 'message' => 'Team leads can only assign Employee role.'], 403);
            }

            $targetUser = User::findOrFail($data['user_id']);
            if (!$this->canManageUser($currentUser, $targetUser)) {
                return response()->json(['success' => false, 'message' => 'You can only manage users in your department.'], 403);
            }
        }

        $user = User::findOrFail($data['user_id']);
        if (!$user->hasRole($data['role'])) {
            $user->assignRole($data['role']);
        }
        return response()->json([
            'success'=>true,
            'message'=>'Role attached.',
            'data'=>['roles'=>$user->refresh()->roles->pluck('name')]
        ]);
    }

    /** Detach role */
    public function detach(Request $request)
    {
        $data = $request->validate([
            'user_id' => ['required','exists:users,id'],
            'role'    => ['required','exists:roles,name'],
        ]);
        $user = User::findOrFail($data['user_id']);

        // حماية: ماينفعش تشيل Admin من نفسك
        if ($user->id === $request->user()->id && $data['role']==='Admin') {
            return response()->json(['success'=>false,'message'=>'You cannot remove your own Admin role.'], 422);
        }

        if ($user->hasRole($data['role'])) {
            $user->removeRole($data['role']);
        }
        return response()->json([
            'success'=>true,
            'message'=>'Role detached.',
            'data'=>['roles'=>$user->refresh()->roles->pluck('name')]
        ]);
    }

    /** Bulk attach */
    public function bulkAttach(Request $request)
    {
        $data = $request->validate([
            'user_ids'   => ['required','array','min:1'],
            'user_ids.*' => ['integer','exists:users,id'],
            'role'       => ['required','exists:roles,name'],
        ]);
        $affected = 0;
        DB::transaction(function() use (&$affected, $data) {
            $users = User::whereIn('id', $data['user_ids'])->get();
            foreach ($users as $u) {
                if (!$u->hasRole($data['role'])) { $u->assignRole($data['role']); $affected++; }
            }
        });
        return response()->json(['success'=>true,'message'=>"Attached to {$affected} users"]);
    }

    /** Bulk detach */
    public function bulkDetach(Request $request)
    {
        $data = $request->validate([
            'user_ids'   => ['required','array','min:1'],
            'user_ids.*' => ['integer','exists:users,id'],
            'role'       => ['required','exists:roles,name'],
        ]);

        if (in_array($request->user()->id, $data['user_ids'] ?? [], true) && $data['role']==='Admin') {
            return response()->json(['success'=>false,'message'=>'You cannot remove your own Admin role.'], 422);
        }

        $affected = 0;
        DB::transaction(function() use (&$affected, $data) {
            $users = User::whereIn('id', $data['user_ids'])->get();
            foreach ($users as $u) {
                if ($u->hasRole($data['role'])) { $u->removeRole($data['role']); $affected++; }
            }
        });
        return response()->json(['success'=>true,'message'=>"Detached from {$affected} users"]);
    }

    private function canManageUser(User $currentUser, User $targetUser)
    {
        if (!$currentUser->hasRole('Team Lead')) {
            return false;
        }

        $managedDepartments = $currentUser->getManagedDepartments()->pluck('id')->toArray();

        $targetEmployee = $targetUser->employee;
        if (!$targetEmployee) {
            return false;
        }

        return in_array($targetEmployee->department_id, $managedDepartments);
    }
}
