<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Department;
use App\Models\Employee;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class TeamLeadController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:Team Lead']);
    }


    public function dashboard(Request $request)
    {
        $currentUser = $request->user();
        $managedDepartments = $currentUser->getManagedDepartments();

        if ($managedDepartments->isEmpty()) {
            return view('pages.team-lead.dashboard', [
                'departmentStats' => [],
                'filter' => 'day',
                'startDate' => now(),
                'endDate' => now(),
                'errorMessage' => 'You are not assigned as team lead to any department.'
            ]);
        }

        // Date filtering logic (same as HR dashboard)
        $filter = $request->get('filter', 'day');
        $customStart = $request->get('custom_start');
        $customEnd = $request->get('custom_end');

        switch ($filter) {
            case 'week':
                $startDate = Carbon::now()->startOfWeek();
                $endDate = Carbon::now()->endOfWeek();
                break;
            case 'month':
                $startDate = Carbon::now()->startOfMonth();
                $endDate = Carbon::now()->endOfMonth();
                break;
            case 'custom':
                $startDate = $customStart ? Carbon::parse($customStart) : Carbon::now();
                $endDate = $customEnd ? Carbon::parse($customEnd) : Carbon::now();
                break;
            default: // day
                $startDate = Carbon::now()->startOfDay();
                $endDate = Carbon::now()->endOfDay();
                break;
        }

        $departmentStats = [];

        foreach ($managedDepartments as $department) {
            // Get all employees in this department
            $employees = Employee::where('department_id', $department->id)
                ->whereHas('user')
                ->with('user')
                ->get();

            $totalEmployees = $employees->count();
            $attendedEmployees = 0;

            if ($totalEmployees > 0) {
                // Count employees who have attendance records in the date range using performance_logs
                $attendedEmployees = 0;

                foreach ($employees as $employee) {
                    if ($employee->employee_code) {
                        $hasAttendance = DB::table('attendance_logs')
                            ->where('employee_code', $employee->employee_code)
                            ->where('type', 'check_in')
                            ->whereBetween('check_time', [$startDate, $endDate])
                            ->exists();

                        if ($hasAttendance) {
                            $attendedEmployees++;
                        }
                    }
                }
            }

            $attendanceRate = $totalEmployees > 0 ? round(($attendedEmployees / $totalEmployees) * 100) : 0;

            $departmentStats[] = [
                'id' => $department->id,
                'name' => $department->name,
                'total_employees' => $totalEmployees,
                'attended_employees' => $attendedEmployees,
                'attendance_rate' => $attendanceRate,
            ];
        }

        return view('pages.team-lead.dashboard', compact(
            'departmentStats',
            'filter',
            'startDate',
            'endDate'
        ));
    }

    public function departmentEmployees(Request $request, $departmentId)
    {

//        dd('here');
        $currentUser = $request->user();

        // Check if user can manage this department
        if (!$currentUser->isTeamLeadOfDepartment($departmentId)) {
            abort(403, 'You can only view employees from your managed departments.');
        }

        $department = Department::findOrFail($departmentId);

        // Date filtering
        $filter = $request->get('filter', 'day');
        $customStart = $request->get('custom_start');
        $customEnd = $request->get('custom_end');

        switch ($filter) {
            case 'week':
                $startDate = Carbon::now()->startOfWeek();
                $endDate = Carbon::now()->endOfWeek();
                break;
            case 'month':
                $startDate = Carbon::now()->startOfMonth();
                $endDate = Carbon::now()->endOfMonth();
                break;
            case 'custom':
                $startDate = $customStart ? Carbon::parse($customStart) : Carbon::now();
                $endDate = $customEnd ? Carbon::parse($customEnd) : Carbon::now();
                break;
            default:
                $startDate = Carbon::now()->startOfDay();
                $endDate = Carbon::now()->endOfDay();
                break;
        }

        // Get employees with attendance data using performance_logs
        $employees = Employee::where('department_id', $departmentId)
            ->whereHas('user')
            ->with(['user', 'department'])
            ->get()
            ->map(function ($employee) use ($startDate, $endDate) {
                $attendanceRecords = collect();
                $attendedDays = 0;
                $latestCheckTime = null;

                if ($employee->employee_code) {
                    // Get attendance data from attendance_logs for accurate datetime values
                    $attendanceRecords = DB::table('attendance_logs')
                        ->where('employee_code', $employee->employee_code)
                        ->whereBetween('check_time', [$startDate, $endDate])
                        ->orderBy('check_time', 'desc')
                        ->get();

                    // Count unique days with attendance (check-ins)
                    $attendedDays = $attendanceRecords
                        ->where('type', 'check_in')
                        ->groupBy(function ($item) {
                            return Carbon::parse($item->check_time)->format('Y-m-d');
                        })
                        ->count();

                    // Get latest check-in time
                    $latestCheckIn = $attendanceRecords
                        ->where('type', 'check_in')
                        ->first();
                    $latestCheckTime = $latestCheckIn ? $latestCheckIn->check_time : null;
                }

                $totalDays = $startDate->diffInDays($endDate) + 1;
                $attendanceRate = $totalDays > 0 ? round(($attendedDays / $totalDays) * 100) : 0;

                return [
                    'id' => $employee->id,
                    'user_id' => $employee->user_id,
                    'name' => $employee->full_name,
                    'email' => $employee->user->email ?? 'N/A',
                    'employee_code' => $employee->employee_code,
                    'attended_days' => $attendedDays,
                    'total_days' => $totalDays,
                    'attendance_rate' => $attendanceRate,
                    'latest_check_in' => $latestCheckTime,
                    'attendance_records' => $attendanceRecords,
                ];
            });

        return view('pages.team-lead.department-employees', compact(
            'employees',
            'department',
            'filter',
            'startDate',
            'endDate'
        ));
    }

    public function employeePerformance(Request $request, $employeeId)
    {
        $employee = Employee::with(['user', 'department'])->findOrFail($employeeId);
        $currentUser = $request->user();

        // Check if user can manage this employee's department
        if (!$currentUser->isTeamLeadOfDepartment($employee->department_id)) {
            abort(403, 'You can only view employees from your managed departments.');
        }

        // Date filtering logic (same as dashboard)
        $filter = $request->get('filter', 'month');
        $customStart = $request->get('custom_start');
        $customEnd = $request->get('custom_end');

        switch ($filter) {
            case 'day':
                $startDate = Carbon::now()->startOfDay();
                $endDate = Carbon::now()->endOfDay();
                break;
            case 'week':
                $startDate = Carbon::now()->startOfWeek();
                $endDate = Carbon::now()->endOfWeek();
                break;
            case 'custom':
                $startDate = $customStart ? Carbon::parse($customStart) : Carbon::now()->subDays(30);
                $endDate = $customEnd ? Carbon::parse($customEnd) : Carbon::now();
                break;
            default: // month
                $startDate = Carbon::now()->startOfMonth();
                $endDate = Carbon::now()->endOfMonth();
                break;
        }

        $attendanceRecords = collect();
        $dailyAttendance = collect();

        if ($employee->employee_code) {
            // Get raw attendance logs (check-in/check-out records)
            $attendanceRecords = DB::table('attendance_logs')
                ->where('employee_code', $employee->employee_code)
                ->whereBetween('check_time', [$startDate, $endDate])
                ->orderBy('check_time', 'desc')
                ->get();

            // Group by date and calculate daily stats
            $dailyAttendance = $attendanceRecords
                ->groupBy(function ($item) {
                    return Carbon::parse($item->check_time)->format('Y-m-d');
                })
                ->map(function ($dayRecords, $date) {
                    $checkIns = $dayRecords->where('type', 'check_in');
                    $checkOuts = $dayRecords->where('type', 'check_out');

                    $firstCheckIn = $checkIns->sortBy('check_time')->first();
                    $lastCheckOut = $checkOuts->sortByDesc('check_time')->first();

                    $workingHours = null;
                    if ($firstCheckIn && $lastCheckOut) {
                        $checkInTime = Carbon::parse($firstCheckIn->check_time);
                        $checkOutTime = Carbon::parse($lastCheckOut->check_time);
                        $workingHours = $checkOutTime->diffInHours($checkInTime, true);
                    }

                    return [
                        'date' => $date,
                        'check_in' => $firstCheckIn ? Carbon::parse($firstCheckIn->check_time)->format('H:i') : null,
                        'check_out' => $lastCheckOut ? Carbon::parse($lastCheckOut->check_time)->format('H:i') : null,
                        'check_in_full' => $firstCheckIn ? Carbon::parse($firstCheckIn->check_time) : null,
                        'check_out_full' => $lastCheckOut ? Carbon::parse($lastCheckOut->check_time) : null,
                        'working_hours' => $workingHours,
                        'total_records' => $dayRecords->count(),
                        'check_ins' => $checkIns->count(),
                        'check_outs' => $checkOuts->count(),
                    ];
                })
                ->sortByDesc('date')
                ->values();
        }

        // Calculate performance metrics
        $totalWorkDays = $startDate->diffInDays($endDate) + 1;
        $attendedDays = $dailyAttendance->where('check_in', '!=', null)->count();
        $attendanceRate = $totalWorkDays > 0 ? round(($attendedDays / $totalWorkDays) * 100) : 0;

        $avgCheckInTime = null;
        $avgWorkHours = null;

        if ($dailyAttendance->count() > 0) {
            // Calculate average check-in time
            $checkInTimes = $dailyAttendance->where('check_in', '!=', null)->pluck('check_in_full');
            if ($checkInTimes->count() > 0) {
                $totalMinutes = $checkInTimes->sum(function ($time) {
                    return $time->hour * 60 + $time->minute;
                });
                $avgMinutes = $totalMinutes / $checkInTimes->count();
                $avgCheckInTime = sprintf('%02d:%02d', floor($avgMinutes / 60), $avgMinutes % 60);
            }

            // Calculate average working hours
            $workingHours = $dailyAttendance->where('working_hours', '!=', null)->pluck('working_hours');
            if ($workingHours->count() > 0) {
                $avgWorkHours = round($workingHours->avg(), 1);
            }
        }

        return view('pages.team-lead.employee-performance', compact(
            'employee',
            'attendanceRecords',
            'dailyAttendance',
            'attendanceRate',
            'attendedDays',
            'totalWorkDays',
            'avgCheckInTime',
            'avgWorkHours',
            'startDate',
            'endDate',
            'filter'
        ));
    }

    public function employeeExport(Request $request, $employeeId)
    {
        $employee = Employee::with(['user', 'department'])->findOrFail($employeeId);
        $currentUser = $request->user();

        // Check if user can manage this employee's department
        if (!$currentUser->isTeamLeadOfDepartment($employee->department_id)) {
            abort(403, 'You can only export data for employees from your managed departments.');
        }

        // Date filtering logic (same as employeePerformance)
        $filter = $request->get('filter', 'month');
        $customStart = $request->get('custom_start');
        $customEnd = $request->get('custom_end');

        switch ($filter) {
            case 'day':
                $startDate = Carbon::now()->startOfDay();
                $endDate = Carbon::now()->endOfDay();
                break;
            case 'week':
                $startDate = Carbon::now()->startOfWeek();
                $endDate = Carbon::now()->endOfWeek();
                break;
            case 'custom':
                $startDate = $customStart ? Carbon::parse($customStart) : Carbon::now()->subDays(30);
                $endDate = $customEnd ? Carbon::parse($customEnd) : Carbon::now();
                break;
            default: // month
                $startDate = Carbon::now()->startOfMonth();
                $endDate = Carbon::now()->endOfMonth();
                break;
        }

        $attendanceRecords = collect();
        $csvData = [];

        if ($employee->employee_code) {
            // Get attendance data from attendance_logs
            $attendanceRecords = DB::table('attendance_logs')
                ->where('employee_code', $employee->employee_code)
                ->whereBetween('check_time', [$startDate, $endDate])
                ->orderBy('check_time', 'desc')
                ->get();

            // Group by date and prepare CSV data
            $dailyAttendance = $attendanceRecords
                ->groupBy(function ($item) {
                    return Carbon::parse($item->check_time)->format('Y-m-d');
                })
                ->map(function ($dayRecords, $date) {
                    $checkIns = $dayRecords->where('type', 'check_in');
                    $checkOuts = $dayRecords->where('type', 'check_out');

                    $firstCheckIn = $checkIns->sortBy('check_time')->first();
                    $lastCheckOut = $checkOuts->sortByDesc('check_time')->first();

                    $workingHours = null;
                    if ($firstCheckIn && $lastCheckOut) {
                        $checkInTime = Carbon::parse($firstCheckIn->check_time);
                        $checkOutTime = Carbon::parse($lastCheckOut->check_time);
                        $workingHours = $checkOutTime->diffInHours($checkInTime, true);
                    }

                    return [
                        'date' => $date,
                        'check_in' => $firstCheckIn ? Carbon::parse($firstCheckIn->check_time)->format('H:i:s') : '',
                        'check_out' => $lastCheckOut ? Carbon::parse($lastCheckOut->check_time)->format('H:i:s') : '',
                        'working_hours' => $workingHours ? number_format($workingHours, 2) : '',
                        'check_ins_count' => $checkIns->count(),
                        'check_outs_count' => $checkOuts->count(),
                        'total_records' => $dayRecords->count(),
                    ];
                })
                ->sortBy('date')
                ->values();

            // Prepare CSV data
            $csvData[] = [
                'Employee Name',
                'Employee Code',
                'Department',
                'Date',
                'Day',
                'Check In',
                'Check Out',
                'Working Hours',
                'Check-ins Count',
                'Check-outs Count',
                'Total Records'
            ];

            foreach ($dailyAttendance as $record) {
                $csvData[] = [
                    $employee->full_name,
                    $employee->employee_code,
                    $employee->department->name ?? 'N/A',
                    Carbon::parse($record['date'])->format('Y-m-d'),
                    Carbon::parse($record['date'])->format('l'),
                    $record['check_in'],
                    $record['check_out'],
                    $record['working_hours'],
                    $record['check_ins_count'],
                    $record['check_outs_count'],
                    $record['total_records']
                ];
            }
        }

        // Generate filename
        $filename = sprintf(
            'employee_attendance_%s_%s_to_%s.csv',
            str_replace(' ', '_', strtolower($employee->full_name)),
            $startDate->format('Y-m-d'),
            $endDate->format('Y-m-d')
        );

        // Create CSV response
        $callback = function() use ($csvData) {
            $file = fopen('php://output', 'w');
            foreach ($csvData as $row) {
                fputcsv($file, $row);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    public function departmentExport(Request $request, $departmentId)
    {
        $currentUser = $request->user();

        // Check if user can manage this department
        if (!$currentUser->isTeamLeadOfDepartment($departmentId)) {
            abort(403, 'You can only export data for departments you manage.');
        }

        $department = Department::findOrFail($departmentId);

        // Date filtering logic (same as dashboard)
        $filter = $request->get('filter', 'day');
        $customStart = $request->get('custom_start');
        $customEnd = $request->get('custom_end');

        switch ($filter) {
            case 'week':
                $startDate = Carbon::now()->startOfWeek();
                $endDate = Carbon::now()->endOfWeek();
                break;
            case 'month':
                $startDate = Carbon::now()->startOfMonth();
                $endDate = Carbon::now()->endOfMonth();
                break;
            case 'custom':
                $startDate = $customStart ? Carbon::parse($customStart) : Carbon::now();
                $endDate = $customEnd ? Carbon::parse($customEnd) : Carbon::now();
                break;
            default: // day
                $startDate = Carbon::now()->startOfDay();
                $endDate = Carbon::now()->endOfDay();
                break;
        }

        // Get employees with attendance data
        $employees = Employee::where('department_id', $departmentId)
            ->whereHas('user')
            ->with(['user', 'department'])
            ->get();

        $csvData = [];

        // CSV Headers
        $csvData[] = [
            'Employee Name',
            'Employee Code',
            'Email',
            'Department',
            'Date',
            'Day',
            'Check In',
            'Check Out',
            'Working Hours',
            'Check-ins Count',
            'Check-outs Count',
            'Total Records'
        ];

        foreach ($employees as $employee) {
            if ($employee->employee_code) {
                // Get attendance data from attendance_logs
                $attendanceRecords = DB::table('attendance_logs')
                    ->where('employee_code', $employee->employee_code)
                    ->whereBetween('check_time', [$startDate, $endDate])
                    ->orderBy('check_time', 'desc')
                    ->get();

                // Group by date and prepare data
                $dailyAttendance = $attendanceRecords
                    ->groupBy(function ($item) {
                        return Carbon::parse($item->check_time)->format('Y-m-d');
                    })
                    ->map(function ($dayRecords, $date) {
                        $checkIns = $dayRecords->where('type', 'check_in');
                        $checkOuts = $dayRecords->where('type', 'check_out');

                        $firstCheckIn = $checkIns->sortBy('check_time')->first();
                        $lastCheckOut = $checkOuts->sortByDesc('check_time')->first();

                        $workingHours = null;
                        if ($firstCheckIn && $lastCheckOut) {
                            $checkInTime = Carbon::parse($firstCheckIn->check_time);
                            $checkOutTime = Carbon::parse($lastCheckOut->check_time);
                            $workingHours = $checkOutTime->diffInHours($checkInTime, true);
                        }

                        return [
                            'date' => $date,
                            'check_in' => $firstCheckIn ? Carbon::parse($firstCheckIn->check_time)->format('H:i:s') : '',
                            'check_out' => $lastCheckOut ? Carbon::parse($lastCheckOut->check_time)->format('H:i:s') : '',
                            'working_hours' => $workingHours ? number_format($workingHours, 2) : '',
                            'check_ins_count' => $checkIns->count(),
                            'check_outs_count' => $checkOuts->count(),
                            'total_records' => $dayRecords->count(),
                        ];
                    })
                    ->sortBy('date');

                // Add data for each day or at least one row per employee
                if ($dailyAttendance->count() > 0) {
                    foreach ($dailyAttendance as $record) {
                        $csvData[] = [
                            $employee->full_name,
                            $employee->employee_code,
                            $employee->user->email ?? 'N/A',
                            $department->name,
                            Carbon::parse($record['date'])->format('Y-m-d'),
                            Carbon::parse($record['date'])->format('l'),
                            $record['check_in'],
                            $record['check_out'],
                            $record['working_hours'],
                            $record['check_ins_count'],
                            $record['check_outs_count'],
                            $record['total_records']
                        ];
                    }
                } else {
                    // Add row even if no attendance data
                    $csvData[] = [
                        $employee->full_name,
                        $employee->employee_code,
                        $employee->user->email ?? 'N/A',
                        $department->name,
                        '',
                        '',
                        'No Data',
                        'No Data',
                        '',
                        0,
                        0,
                        0
                    ];
                }
            }
        }

        // Generate filename
        $filename = sprintf(
            'department_attendance_%s_%s_to_%s.csv',
            str_replace(' ', '_', strtolower($department->name)),
            $startDate->format('Y-m-d'),
            $endDate->format('Y-m-d')
        );

        // Create CSV response
        $callback = function() use ($csvData) {
            $file = fopen('php://output', 'w');
            foreach ($csvData as $row) {
                fputcsv($file, $row);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }
}
