<?php

namespace App\Http\Controllers;

use App\Models\Employee;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;
use Throwable;

class ToolsController extends Controller
{
    /**
     * Sync Employees -> Users
     *
     * POST /admin/tools/sync-employees-to-users
     *
     * Middleware: ['auth', 'can:admin.tools']
     * Optional payload:
     *  - dry_run (bool)           : لا يكتب في الداتابيز، بس بيرجع تقرير
     *  - chunk (int, default 200) : حجم الدُفعة في المعالجة
     *  - default_password (string): باسورد للمستخدمين الجدد (افتراضي "password")
     *
     * Response (JSON): summary counts + log
     */
    public function syncEmployeesToUsers(Request $request)
    {
        // تأكيد التفويض كطبقة إضافية بجانب الميدلوير
        Gate::authorize('admin.tools');

        $dryRun          = (bool) $request->boolean('dry_run', false);
        $chunkSize       = (int)  max(50, (int) $request->input('chunk', 200));
        $defaultPassword = (string) ($request->input('default_password', 'password'));

        $summary = [
            'dry_run'        => $dryRun,
            'created_users'  => 0,
            'linked_users'   => 0, // للاستخدام لاحقاً لو هتربط بيوزر موجود عن طريق إيميل مثلاً
            'skipped'        => 0,
            'errors'         => 0,
            'log'            => [],
        ];

        try {
            // ملاحظة: من الأفضل عدم لفّ كل العملية في معاملة واحدة كبيرة.
            // هنستخدم معاملات صغيرة لكل دفعة لتقليل الـ locks.
            Employee::orderBy('id')
                ->chunkById($chunkSize, function ($employees) use (&$summary, $dryRun, $defaultPassword) {
                    DB::beginTransaction();
                    try {
                        foreach ($employees as $emp) {
                            // لو الموظف متربط أصلاً بيوزر — نعدّي
                            if (!empty($emp->user_id)) {
                                $summary['skipped']++;
                                continue;
                            }

                            // تجهيز الاسم
                            $first = trim($emp->first_name ?? '');
                            $last  = trim($emp->last_name ?? '');
                            $name  = trim($first . ' ' . $last);
                            if ($name === '') {
                                $name = $emp->name ?? ('Employee #' . $emp->id);
                            }

                            // في الوضع التجريبي: سجل وبس
                            if ($dryRun) {
                                $summary['log'][] = "[DRY] Would create USER for EMP#{$emp->id} ({$name})";
                                continue;
                            }

                            // إنشاء User جديد
                            $user = User::create([
                                'name'     => $name,
                                'password' => Hash::make($defaultPassword),
                            ]);

                            // ربط الموظف باليوزر
                            $emp->user_id = $user->id;
                            $emp->save();

                            $summary['created_users']++;
                            $summary['log'][] = "Created USER#{$user->id} for EMP#{$emp->id} ({$name})";
                        }

                        DB::commit();
                    } catch (Throwable $e) {
                        DB::rollBack();
                        $summary['errors']++;
                        $summary['log'][] = 'BATCH ERROR: ' . $e->getMessage();
                        // بنكمل في الدفعات التالية بدل ما نوقف كله
                    }
                });

            $status = $summary['errors'] > 0 ? 207 : 200; // 207: Multi-Status (جزء نجح/جزء فشل)
            return response()->json($summary, $status);
        } catch (Throwable $e) {
            $summary['errors']++;
            $summary['log'][] = 'FATAL: ' . $e->getMessage();
            return response()->json($summary, 500);
        }
    }
}
