<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Department;
use Illuminate\Http\Request;

class UserController extends Controller
{
    public function index(Request $request)
    {
        $query = User::query()
            ->with(['employee.department'])
            ->whereHas('employee');

        // 🔍 بحث بالاسم (عربي/إنجليزي) + دعم التباين في الحروف
        if ($search = $request->input('search')) {
            $normalized = $this->normalizeArabic($search);
            $query->where(function ($q) use ($normalized) {
                $q->whereRaw("LOWER(REPLACE(REPLACE(REPLACE(REPLACE(name, 'ي', 'ى'), 'ى', 'ي'), 'ة', 'ه'), 'أ', 'ا')) LIKE ?", ["%{$normalized}%"])
                    ->orWhereHas('employee', function ($sub) use ($normalized) {
                        $sub->whereRaw("LOWER(REPLACE(REPLACE(REPLACE(REPLACE(first_name, 'ي', 'ى'), 'ى', 'ي'), 'ة', 'ه'), 'أ', 'ا')) LIKE ?", ["%{$normalized}%"])
                            ->orWhereRaw("LOWER(REPLACE(REPLACE(REPLACE(REPLACE(last_name, 'ي', 'ى'), 'ى', 'ي'), 'ة', 'ه'), 'أ', 'ا')) LIKE ?", ["%{$normalized}%"]);
                    });
            });
        }

        // 🔽 فلترة حسب القسم
        if ($department = $request->input('department_id')) {
            $query->whereHas('employee', fn($q) => $q->where('department_id', $department));
        }

        // 📧 فلترة حسب وجود الإيميل
        if ($request->input('email_status') === 'missing') {
            $query->whereNull('email')->orWhere('email', '');
        } elseif ($request->input('email_status') === 'has') {
            $query->whereNotNull('email')->where('email', '<>', '');
        }

        // 🟢 فلترة الحالة
        if ($status = $request->input('status')) {
            $query->where('status', $status === 'active' ? 1 : 0);
        }

        $users = $query->paginate(20)->appends($request->query());
        $departments = Department::active()->get(['id', 'name']);

        // AJAX response
        if ($request->ajax()) {
            return view('hr.users.partials.table', compact('users'))->render();
        }

        return view('hr.users.index', compact('users', 'departments'));
    }

    // === Helper function to normalize Arabic characters ===
    private function normalizeArabic($text)
    {
        $map = [
            'أ' => 'ا',
            'إ' => 'ا',
            'آ' => 'ا',
            'ى' => 'ي',
            'ئ' => 'ي',
            'ؤ' => 'و',
            'ة' => 'ه',
        ];
        return strtr($text, $map);
    }

    public function edit(User $user)
    {
        $departments = Department::active()->get(['id', 'name']);
        return view('hr.users.edit', compact('user', 'departments'));
    }

    public function update(Request $request, User $user)
    {
        $request->validate([
            'email' => 'nullable|email|unique:users,email,' . $user->id,
            'department_id' => 'nullable|exists:departments,id',
        ]);

        // تحديث بيانات المستخدم
        if ($request->filled('email')) {
            $user->email = $request->email;
        }
        if ($request->filled('name')) {
            $user->name = $request->name;
        }

        $user->save();

        // تحديث القسم المرتبط بالموظف
        if ($request->filled('department_id') && $user->employee) {
            $user->employee->update(['department_id' => $request->department_id]);
        }

        return redirect()->route('hr.users.index')->with('success', 'User info updated successfully.');
    }
}
