<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Http\Requests\Vacation\StoreVacationRequest;
use App\Http\Requests\Vacation\UpdateVacationRequest;
use App\Models\User;
use App\Models\Vacation;
use App\Models\VacationType;
use App\Services\VacationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;

// أحداث البث (Echo + Pusher)
use App\Events\VacationRequested;
use App\Events\VacationStatusChanged;

// حفظ التاريخ في جدول notifications
use App\Notifications\VacationStatusNotification;

class VacationController extends Controller
{
    public function __construct(private VacationService $service) {}

    public function dt(Request $request)
    {
        // ===== Inputs (تطبيع آمن)
        $searchRaw = $request->input('search', '');
        $search = is_array($searchRaw)
            ? trim((string)($searchRaw['value'] ?? ''))
            : trim((string)$searchRaw);

        $typeId = (string) $request->input('type', '');
        $status = (string) $request->input('status', '');

        // ترتيب
        // الأعمدة في الجدول: [0 id][1 employee_name][2 type_name][3 start][4 end][5 total][6 status][7 actions]
        $orderColIdx = (int) $request->input('order.0.column', 0);
        $orderDir    = strtolower((string) $request->input('order.0.dir', 'desc')) === 'asc' ? 'asc' : 'desc';
        $map = [
            0 => 'vacations.id',
            1 => 'employee_name',         // يحتاج join
            2 => 'vacation_types.name',
            3 => 'vacations.start_date',
            4 => 'vacations.end_date',
            5 => 'vacations.total_days',
            6 => 'vacations.status',
            7 => null, // actions
        ];
        $orderBy = $map[$orderColIdx] ?? 'vacations.id';

        // Pagination
        $start  = max(0, (int) $request->input('start', 0));
        $length = (int) $request->input('length', 10);
        if ($length <= 0 || $length > 1000) $length = 10;

        // ===== Base Query
        $base = Vacation::query()->with(['employee.user', 'type']);

        // ===== رؤية حسب الدور (بدون الاعتماد على can('viewAny')):
        $user         = $request->user();
        $employee     = $user->employee; // ممكن تكون null
        $isAdminLike  = $user->hasAnyRole(['Admin', 'Manager', 'HR']);
        $isTeamLeader = $user->hasRole('Team Lead');

        if ($isAdminLike) {
            // يشوف الكل
        } elseif ($isTeamLeader) {
            // يشوف سجلاته + كل موظفي نفس القسم
            $leaderDeptId = optional($employee)->department_id;
            abort_if(!$leaderDeptId, 403, 'لا يوجد قسم مرتبط بالتيم ليدر.');

            $base->where(function ($w) use ($employee, $leaderDeptId) {
                $w->where('employee_id', optional($employee)->id)
                    ->orWhereIn('employee_id', function ($sub) use ($leaderDeptId) {
                        $sub->select('id')
                            ->from('employees')
                            ->where('department_id', $leaderDeptId);
                    });
            });
        } else {
            // Employee: سجلاته فقط
            $employeeId = optional($employee)->id;
            abort_if(!$employeeId, 403, 'لا توجد بطاقة موظف مرتبطة بالحساب.');
            $base->where('employee_id', $employeeId);
        }

        // ===== فلاتر النوع والحالة (تدعم Enum أو نص)
        $base->when($typeId !== '', fn($q) => $q->where('vacation_type_id', $typeId));

        $base->when($status !== '', function ($q) use ($status) {
            $normalized = strtolower(trim($status));

            if (enum_exists(\App\Enums\VacationStatus::class)) {
                $case = \App\Enums\VacationStatus::tryFrom($status)
                    ?? \App\Enums\VacationStatus::tryFrom($normalized);

                if (!$case) {
                    foreach (\App\Enums\VacationStatus::cases() as $c) {
                        if (strtolower($c->name) === $normalized) {
                            $case = $c;
                            break;
                        }
                    }
                }

                if ($case) {
                    $q->where('status', $case->value ?? $case->name);
                    return;
                }
            }

            // لو مش Enum/أو مدخلة كنص: قارن case-insensitive
            $q->whereRaw('LOWER(status) = ?', [$normalized]);
        });

        // إجمالي قبل البحث العام
        $recordsTotal = (clone $base)->count();

        // ===== البحث العام (اسم الموظف / اسم النوع / سبب)
        $filtered = (clone $base);
        if ($search !== '') {
            $filtered->where(function ($w) use ($search) {
                $w->whereHas('employee', function ($e) use ($search) {
                    $e->where('first_name', 'like', "%{$search}%")
                        ->orWhere('last_name', 'like', "%{$search}%")
                        ->orWhereHas('user', function ($u) use ($search) {
                            $u->where('name', 'like', "%{$search}%");
                        });
                })->orWhereHas('type', function ($t) use ($search) {
                    $t->where('name', 'like', "%{$search}%");
                })->orWhere('member_reason', 'like', "%{$search}%");
            });
        }

        $recordsFiltered = (clone $filtered)->count();

        // ===== ترتيب (مع الـjoins لو لزم)
        if ($orderBy === 'employee_name') {
            $filtered->leftJoin('employees', 'employees.id', '=', 'vacations.employee_id')
                ->orderBy('employees.first_name', $orderDir)
                ->orderBy('employees.last_name',  $orderDir)
                ->select('vacations.*');
        } elseif ($orderBy === 'vacation_types.name') {
            $filtered->leftJoin('vacation_types', 'vacation_types.id', '=', 'vacations.vacation_type_id')
                ->orderBy('vacation_types.name', $orderDir)
                ->select('vacations.*');
        } else {
            $filtered->orderBy($orderBy, $orderDir);
        }

        // ===== Fetch
        $rows = $filtered->skip($start)->take($length)->get();

        // ===== Transform
        $authUser = auth()->user();
        $now = now();
        $data = $rows->map(function (Vacation $v) use ($authUser, $now) {
            $empName     = $v->employee?->full_name ?: ($v->employee?->user?->name ?? '—');
            $statusValue = $v->status?->value ?? (string) $v->status;

            // منطق الحالة النهائية + نافذة HR
            $isFinal = in_array($statusValue, [
                'hr_approved',
                'leader_rejected',
                'hr_rejected',
                'cancelled_by_member'
            ], true) || ($statusValue === 'leader_approved' && !$v->needs_hr);

            $hrWindowOpen = $now->lt(optional($v->start_date)->copy()->startOfDay());

            return [
                'id'                  => $v->id,
                'employee_name'       => $empName,
                'type_name'           => $v->type?->name ?? '—',
                'start_date'          => optional($v->start_date)->format('Y-m-d'),
                'end_date'            => optional($v->end_date)->format('Y-m-d'),
                'total_days'          => $v->total_days,
                'status'              => $statusValue,
                'status_value'        => $statusValue,

                // === صلاحيات الأزرار حسب الـPolicy ===
                'can_update'          => $authUser?->can('update', $v) ?? false,         // Owner + Pending فقط
                'can_delete'          => $authUser?->can('delete', $v) ?? false,         // Owner + Pending فقط (إلغاء منطقي)
                'can_leader_approve'  => $authUser?->can('leaderApprove', $v) ?? false,  // TL فقط ونفس القسم
                'can_hr_approve'      => $authUser?->can('hrApprove', $v) ?? false,      // HR/Admin فقط (HR مش لطلبه)
                'can_reject'          => $authUser?->can('reject', $v) ?? false,         // TL (نفس القسم وغير مالك) أو HR/Admin

                // أعلام مساعدة للـUI
                'needs_hr'            => (bool)$v->needs_hr,
                'is_final'            => $isFinal,
                'hr_window_open'      => $hrWindowOpen,
            ];
        });

        return response()->json([
            'draw'            => (int) $request->input('draw'),
            'recordsTotal'    => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data'            => $data,
        ]);
    }

    public function index(Request $request)
    {
        $this->authorize('viewAny', Vacation::class);
        $types = VacationType::orderBy('name')->get(['id', 'name']);
        return view('pages.vacations.index', compact('types'));
    }

    public function create()
    {
        return view('pages.vacations.create', [
            'types' => VacationType::orderBy('name')->get()
        ]);
    }

    public function store(StoreVacationRequest $request)
    {
        $this->authorize('create', Vacation::class);
        $employee = $request->user()->employee;

        $vac = $this->service->create(
            data: $request->validated(),
            employeeId: $employee->id,
            requestedByUserId: $request->user()->id
        )->load(['employee.user', 'type']);

        // 1) بث Echo (Realtime)
        $recipientIds = $this->resolveRecipientsForNewRequest($vac);
        event(new VacationRequested($vac, $recipientIds));

        // 2) حفظ History في جدول notifications
        $this->notifyUsers($recipientIds, new VacationStatusNotification(
            vacation: $vac,
            action: 'submitted',
            actor: $request->user()
        ));

        $msg = 'تم إنشاء طلب الإجازة بنجاح.';

        return $request->expectsJson()
            ? response()->json([
                'ok' => true,
                'id' => $vac->id,
                'message' => $msg,
                'toast' => ['variant' => 'success', 'title' => 'نجاح', 'body' => $msg],
            ])
            : redirect()->route('vacations.show', $vac)->with('success', $msg);
    }

    public function show(Vacation $vacation)
    {
        $this->authorize('view', $vacation);
        return view('pages.vacations.show', [
            'vacation' => $vacation->load('employee.user', 'type')
        ]);
    }

    public function edit(Vacation $vacation)
    {
        $this->authorize('update', $vacation);
        return view('pages.vacations.edit', [
            'vacation' => $vacation->load('employee.user', 'type'),
            'types'    => VacationType::orderBy('name')->get()
        ]);
    }

    public function update(UpdateVacationRequest $request, Vacation $vacation)
    {
        $this->authorize('update', $vacation);
        $this->service->update($vacation, $request->validated());

        $msg = 'تم تحديث طلب الإجازة.';

        return $request->expectsJson()
            ? response()->json([
                'ok' => true,
                'id' => $vacation->id,
                'message' => $msg,
                'toast' => ['variant' => 'success', 'title' => 'نجاح', 'body' => $msg],
            ])
            : redirect()->route('vacations.show', $vacation)->with('success', $msg);
    }

    /**
     * إلغاء منطقي بواسطة صاحب الطلب (بديل عن الحذف الفعلي).
     * يعتمد على Policy: delete => Owner + Pending.
     */
    public function destroy(Request $request, Vacation $vacation)
    {
        $this->authorize('delete', $vacation);

        $this->service->cancelByMember($vacation);
        $vacation->refresh()->load(['employee.user', 'type']);

        // بث "إلغاء بواسطة العضو"
        $recipientIds = $this->resolveRecipientsForStatusChange($vacation, actorIsHr: false, actorIsLeader: false);
        event(new VacationStatusChanged(
            vacation: $vacation,
            status: 'cancelled_by_member',
            actorName: $request->user()->name,
            recipientIds: $recipientIds
        ));

        // حفظ History
        $this->notifyUsers($recipientIds, new VacationStatusNotification(
            vacation: $vacation,
            action: 'cancelled',
            actor: $request->user()
        ));

        $msg = 'تم إلغاء طلب الإجازة.';

        return $request->expectsJson()
            ? response()->json([
                'ok' => true,
                'id' => $vacation->id,
                'message' => $msg,
                'toast' => ['variant' => 'success', 'title' => 'تم الإلغاء', 'body' => $msg],
            ])
            : redirect()->route('vacations.index')->with('success', $msg);
    }

    // اعتماد TL
    public function leaderApprove(Request $request, Vacation $vacation)
    {
        $this->authorize('leaderApprove', $vacation);

        $this->service->leaderApprove(
            $vacation,
            reason: (string) $request->string('reason'),
            approverUserId: $request->user()->id
        );

        $vacation->refresh()->load(['employee.user', 'type']);

        $recipientIds = $this->resolveRecipientsForStatusChange($vacation, actorIsHr: false, actorIsLeader: true);
        event(new VacationStatusChanged(
            vacation: $vacation,
            status: 'approved', // للـ Front: success
            actorName: $request->user()->name,
            recipientIds: $recipientIds
        ));

        $this->notifyUsers($recipientIds, new VacationStatusNotification(
            vacation: $vacation,
            action: 'leader_approved',
            actor: $request->user()
        ));

        $msg = 'تم اعتماد التيم ليدر.';

        return $request->expectsJson()
            ? response()->json([
                'ok'     => true,
                'status' => $vacation->fresh()->status->value ?? (string)$vacation->fresh()->status,
                'message' => $msg,
                'toast'  => ['variant' => 'success', 'title' => 'نجاح', 'body' => $msg],
            ])
            : back()->with('success', $msg);
    }

    // اعتماد HR
    public function hrApprove(Request $request, Vacation $vacation)
    {
        $this->authorize('hrApprove', $vacation);

        $this->service->hrApprove(
            $vacation,
            reason: (string) $request->string('reason'),
            approverUserId: $request->user()->id
        );

        $vacation->refresh()->load(['employee.user', 'type']);

        $recipientIds = $this->resolveRecipientsForStatusChange($vacation, actorIsHr: true, actorIsLeader: false);
        event(new VacationStatusChanged(
            vacation: $vacation,
            status: 'approved', // للـ Front: success
            actorName: $request->user()->name,
            recipientIds: $recipientIds
        ));

        $this->notifyUsers($recipientIds, new VacationStatusNotification(
            vacation: $vacation,
            action: 'hr_approved',
            actor: $request->user()
        ));

        $msg = 'تم اعتماد HR.';

        return $request->expectsJson()
            ? response()->json([
                'ok'     => true,
                'status' => $vacation->fresh()->status->value ?? (string)$vacation->fresh()->status,
                'message' => $msg,
                'toast'  => ['variant' => 'success', 'title' => 'نجاح', 'body' => $msg],
            ])
            : back()->with('success', $msg);
    }

    // رفض (TL أو HR)
    public function reject(Request $request, Vacation $vacation)
    {
        $this->authorize('reject', $vacation);

        $by = $request->user()->hasAnyRole(['HR', 'Admin']) ? 'hr' : 'leader';

        $this->service->reject(
            $vacation,
            reason: (string) $request->string('reason'),
            by: $by,
            approverUserId: $request->user()->id
        );

        $vacation->refresh()->load(['employee.user', 'type']);

        $isHr = $by === 'hr';
        $recipientIds = $this->resolveRecipientsForStatusChange($vacation, actorIsHr: $isHr, actorIsLeader: !$isHr);

        event(new VacationStatusChanged(
            vacation: $vacation,
            status: 'rejected', // للـ Front: error
            actorName: $request->user()->name,
            recipientIds: $recipientIds
        ));

        $this->notifyUsers($recipientIds, new VacationStatusNotification(
            vacation: $vacation,
            action: $isHr ? 'hr_rejected' : 'leader_rejected',
            actor: $request->user()
        ));

        $msg = $by === 'leader' ? 'تم رفض الطلب (TL).' : 'تم رفض الطلب (HR).';

        return $request->expectsJson()
            ? response()->json([
                'ok'     => true,
                'status' => $vacation->fresh()->status->value ?? (string)$vacation->fresh()->status,
                'message' => $msg,
                'toast'  => ['variant' => 'success', 'title' => 'تم التنفيذ', 'body' => $msg],
            ])
            : back()->with('success', $msg);
    }

    /* ================= Helpers: recipients + notify ================ */

    private function notifyUsers(array $recipientIds, \Illuminate\Notifications\Notification $notification): void
    {
        if (empty($recipientIds)) return;
        try {
            $users = User::whereIn('id', $recipientIds)->get();
            foreach ($users as $u) {
                try {
                    $u->notify($notification);
                } catch (\Throwable $e) {
                    report($e);
                }
            }
        } catch (\Throwable $e) {
            // نتجاهل الأخطاء علشان ما نكسرش الفلو اللحظي
        }
    }

    /**
     * مستلمو إشعار "طلب جديد": الموظف + تيم ليدر نفس القسم + HR/Admin.
     */
    private function resolveRecipientsForNewRequest(Vacation $vac): array
    {
        $empUserId   = (int) ($vac->employee?->user_id ?? 0);
        $deptId      = (int) ($vac->employee?->department_id ?? 0);

        $leaderIds   = $this->findTeamLeadersByDepartment($deptId);
        $hrIds       = $this->findHrAndAdmins();

        return $this->uniqueIds(array_merge([$empUserId], $leaderIds, $hrIds));
    }

    /**
     * مستلمو إشعار "تغير حالة": دائمًا الموظف،
     * + لو الفاعل TL → أرسل للـ HR/Admin لاستكمال الخطوة،
     * + لو الفاعل HR → أرسل للـ TL (للمعرفة).
     */
    private function resolveRecipientsForStatusChange(Vacation $vac, bool $actorIsHr, bool $actorIsLeader): array
    {
        $empUserId = (int) ($vac->employee?->user_id ?? 0);
        $deptId    = (int) ($vac->employee?->department_id ?? 0);

        $base = [$empUserId];

        if ($actorIsLeader) {
            $base = array_merge($base, $this->findHrAndAdmins());
        } elseif ($actorIsHr) {
            $base = array_merge($base, $this->findTeamLeadersByDepartment($deptId));
        }

        return $this->uniqueIds($base);
    }

    /**
     * جلب التيم ليدر لنفس القسم (يدعم أكثر من طريقة حسب علاقاتك).
     */
    private function findTeamLeadersByDepartment(int $departmentId): array
    {
        if ($departmentId <= 0) return [];

        $ids = [];

        // 1) لو في علاقة user->leaderOfDepartments (Pivot)
        try {
            $ids = array_merge($ids, User::role('Team Lead')
                ->whereHas('leaderOfDepartments', fn($q) => $q->where('departments.id', $departmentId))
                ->pluck('id')->all());
        } catch (\Throwable $e) {
        }

        // 2) لو التيم ليدر نفسه Employee ومربوط بنفس القسم
        try {
            $ids = array_merge($ids, User::role('Team Lead')
                ->whereHas('employee', fn($q) => $q->where('department_id', $departmentId))
                ->pluck('id')->all());
        } catch (\Throwable $e) {
        }

        return $this->uniqueIds($ids);
    }

    /**
     * HR + Admin users.
     */
    private function findHrAndAdmins(): array
    {
        try {
            return User::role(['HR', 'Admin'])->pluck('id')->all();
        } catch (\Throwable $e) {
            return [];
        }
    }

    /**
     * Unique, filtered ints.
     */
    private function uniqueIds(array $ids): array
    {
        $ids = array_map('intval', array_filter($ids, fn($v) => (int)$v > 0));
        sort($ids);
        return array_values(array_unique($ids));
    }
}
