<?php

namespace App\Http\Controllers\web;

use App\Http\Controllers\Controller;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class AttendanceController extends Controller
{
    public function checkIn(Request $request)
    {
        $request->validate([
            'password' => 'required',
        ]);

        if (! Hash::check($request->password, Auth::user()->password)) {
            return response()->json(['error' => 'Invalid password'], 403);
        }

        // Adjust the time according to the desired offset (e.g., adding 3 hours)
        $now = Carbon::now()->addHours(3);  // Add 3 hours to the current server time

        // Check if the user has already checked in today and hasn't checked out yet
        $existingAttendance = DB::table('_x_test_attendances')
            ->where('user_id', Auth::id())
            ->whereDate('check_in', $now->toDateString())
            ->whereNull('check_out')
            ->first();

        if ($existingAttendance) {
            return response()->json(['error' => 'You have already checked in today and need to check out before you can check in again.'], 403);
        }

        // Create the check-in record
        $attendanceId = DB::table('_x_test_attendances')->insertGetId([
            'user_id' => Auth::id(),
            'check_in' => $now,
            'created_at' => $now,
            'updated_at' => $now,
        ]);

        $attendance = DB::table('_x_test_attendances')->where('id', $attendanceId)->first();

        return response()->json(['attendance' => $attendance]);
    }

    public function checkOut(Request $request)
    {
        $request->validate([
            'password' => 'required',
        ]);

        if (! Hash::check($request->password, Auth::user()->password)) {
            return response()->json(['error' => 'Invalid password'], 403);
        }

        // Adjust the time according to the desired offset (e.g., adding 3 hours)
        $now = Carbon::now()->addHours(3);  // Add 3 hours to the current server time

        // Find the latest attendance record for the user where they haven't checked out
        $attendance = DB::table('_x_test_attendances')
            ->where('user_id', Auth::id())
            ->whereNull('check_out')
            ->orderBy('id', 'desc')
            ->first();

        if (! $attendance) {
            return response()->json(['error' => 'No active check-in found.'], 403);
        }

        // Update the check-out time
        DB::table('_x_test_attendances')
            ->where('id', $attendance->id)
            ->update([
                'check_out' => $now,
                'updated_at' => $now,
            ]);

        $attendance = DB::table('_x_test_attendances')->where('id', $attendance->id)->first();

        return response()->json(['attendance' => $attendance]);
    }

    public function history()
    {
        $attendances = DB::table('_x_test_attendances')
            ->where('user_id', Auth::id())
            ->orderBy('check_in', 'desc')
            ->get();

        return view('common.managment.remote.attendance.history', compact('attendances'));
    }

    public function index()
    {
        return view('attendance.index');
    }

    public function leaderView()
    {

        $startDate = '2024-11-25';

        $tl_type = auth()->user()->type;

        $typeIndex = array_search($tl_type, [
            'Media_Buyer',
            'admin',
            'op',
            'broker',
            'Designer',
            'Sales',
            'Moderator',
            'Content',
        ]);

        if ($typeIndex !== false) {
            $TeamLeads = User::where('type', $typeIndex)->where('sub_type', 1)->get();
            $members = User::where('type', $typeIndex)->where('sub_type', 0)->get();

            // Filter members who have attendance records after start date
            $membersWithAttendance = [];
            foreach ($members as $member) {
                $hasAttendance = DB::table('_x_test_attendances')
                    ->where('user_id', $member->id)
                    ->where('created_at', '>=', $startDate)
                    ->exists();

                if ($hasAttendance) {
                    $membersWithAttendance[] = $member;
                }
            }
            $members = collect($membersWithAttendance);
        } else {
            dd('نوع التيم ليدر غير صالح');
        }

        $events = [];

        foreach ($TeamLeads as $lead) {
            $attendances = DB::table('_x_test_attendances')
                ->where('user_id', $lead->id)
                ->where('created_at', '>=', $startDate)
                ->get();

            foreach ($attendances as $attendance) {
                $events[] = [
                    'title' => $lead->name.' -  In',
                    'start' => $attendance->check_in,
                    'color' => '#2980b9', // Blue for check-in
                ];

                if ($attendance->check_out) {
                    $events[] = [
                        'title' => $lead->name.' - Out',
                        'start' => $attendance->check_out,
                        'color' => '#e74c3c', // Red for check-out
                    ];
                }
            }
        }

        foreach ($members as $member) {
            $attendances = DB::table('_x_test_attendances')
                ->where('user_id', $member->id)
                ->where('created_at', '>=', $startDate)
                ->get();

            foreach ($attendances as $attendance) {
                $events[] = [
                    'title' => $member->name.' -  In',
                    'start' => $attendance->check_in,
                    'color' => '#388813', // Green for check-in
                ];

                if ($attendance->check_out) {
                    $events[] = [
                        'title' => $member->name.' -  Out',
                        'start' => $attendance->check_out,
                        'color' => '#8a2b21', // Red for check-out
                    ];
                }
            }
        }

        return view('common.managment.remote.attendance.all_team_history', compact('events', 'TeamLeads', 'members'));
    }

    public function hrDashboard(Request $request)
    {
        $filter = $request->get('filter', 'day');
        $customStart = $request->get('custom_start');
        $customEnd = $request->get('custom_end');

        // Calculate date range based on filter
        $now = Carbon::now()->addHours(3);
        switch ($filter) {
            case 'week':
                $startDate = $now->copy()->startOfWeek();
                $endDate = $now->copy()->endOfWeek();
                break;
            case 'month':
                $startDate = $now->copy()->startOfMonth();
                $endDate = $now->copy()->endOfMonth();
                break;
            case 'custom':
                $startDate = $customStart ? Carbon::parse($customStart) : $now->copy()->startOfDay();
                $endDate = $customEnd ? Carbon::parse($customEnd) : $now->copy()->endOfDay();
                break;
            default: // day
                $startDate = $now->copy()->startOfDay();
                $endDate = $now->copy()->endOfDay();
        }

        // Get departments that have employees
        $departments = \App\Models\Department::whereHas('employees')->with(['employees.user'])->get();

        $departmentStats = [];

        foreach ($departments as $department) {
            $totalEmployees = $department->employees->count();
            if ($totalEmployees === 0) continue;

            $attendedEmployees = 0;
            $employeeDetails = [];

            foreach ($department->employees as $employee) {
                $attendanceRate = 0;
                $totalAttendances = 0;
                $hasAttendance = false;

                if ($employee->user) {
                    // Get performance from performance_logs using employee_code
                    $performanceRecords = \DB::table('performance_logs')
                        ->where('employee_code', $employee->employee_code)
                        ->whereBetween('check_time', [$startDate, $endDate])
                        ->where('status', '!=', 'absent')
                        ->get();

                    $totalAttendances = $performanceRecords->count();
                    $hasAttendance = $totalAttendances > 0;

                    if ($hasAttendance) {
                        $attendedEmployees++;
                    }

                    $expectedDays = $this->getExpectedWorkDays($startDate, $endDate);
                    $attendanceRate = $expectedDays > 0 ? round(($totalAttendances / $expectedDays) * 100, 2) : 0;
                }

                $employeeDetails[] = [
                    'id' => $employee->id,
                    'name' => $employee->full_name ?? ($employee->user->name ?? 'Unknown'),
                    'attendance_rate' => $attendanceRate,
                    'total_attendances' => $totalAttendances,
                    'has_attendance' => $hasAttendance
                ];
            }

            $departmentAttendanceRate = $totalEmployees > 0 ? round(($attendedEmployees / $totalEmployees) * 100, 2) : 0;

            $departmentStats[] = [
                'id' => $department->id,
                'name' => $department->name,
                'total_employees' => $totalEmployees,
                'attended_employees' => $attendedEmployees,
                'attendance_rate' => $departmentAttendanceRate,
                'employees' => $employeeDetails
            ];
        }

        // Debug information - remove this after testing
        $debugInfo = [
            'total_departments' => $departments->count(),
            'departments_with_employees' => $departments->filter(function($dept) { return $dept->employees->count() > 0; })->count(),
            'total_employees' => $departments->sum(function($dept) { return $dept->employees->count(); }),
            'employees_with_users' => $departments->sum(function($dept) {
                return $dept->employees->filter(function($emp) { return $emp->user !== null; })->count();
            }),
            'date_range' => $startDate->format('Y-m-d') . ' to ' . $endDate->format('Y-m-d'),
            'expected_days' => $this->getExpectedWorkDays($startDate, $endDate)
        ];

        return view('hr.attendance.dashboard', compact('departmentStats', 'filter', 'startDate', 'endDate', 'debugInfo'));
    }

    public function departmentEmployees(Request $request, $departmentId)
    {
        $filter = $request->get('filter', 'day');
        $customStart = $request->get('custom_start');
        $customEnd = $request->get('custom_end');

        // Calculate date range
        $now = Carbon::now()->addHours(3);
        switch ($filter) {
            case 'week':
                $startDate = $now->copy()->startOfWeek();
                $endDate = $now->copy()->endOfWeek();
                break;
            case 'month':
                $startDate = $now->copy()->startOfMonth();
                $endDate = $now->copy()->endOfMonth();
                break;
            case 'custom':
                $startDate = $customStart ? Carbon::parse($customStart) : $now->copy()->startOfDay();
                $endDate = $customEnd ? Carbon::parse($customEnd) : $now->copy()->endOfDay();
                break;
            default:
                $startDate = $now->copy()->startOfDay();
                $endDate = $now->copy()->endOfDay();
        }

        $department = \App\Models\Department::with(['employees.user'])->findOrFail($departmentId);

        $employees = [];
        foreach ($department->employees as $employee) {
            $attendanceCount = 0;
            $recentAttendances = collect();

            if ($employee->user) {
                // Get performance logs for this employee
                $performanceRecords = \DB::table('performance_logs')
                    ->where('employee_code', $employee->employee_code)
                    ->whereBetween('check_time', [$startDate, $endDate])
                    ->orderBy('check_time', 'desc')
                    ->get();

                $attendanceCount = $performanceRecords->where('status', '!=', 'absent')->count();
                $recentAttendances = $performanceRecords->take(5);
            }

            $expectedDays = $this->getExpectedWorkDays($startDate, $endDate);
            $attendanceRate = $expectedDays > 0 ? round(($attendanceCount / $expectedDays) * 100, 2) : 0;

            $employees[] = [
                'id' => $employee->id,
                'name' => $employee->full_name ?? ($employee->user->name ?? 'Unknown'),
                'attendance_rate' => $attendanceRate,
                'total_attendances' => $attendanceCount,
                'recent_attendances' => $recentAttendances
            ];
        }

        return view('hr.attendance.department-employees', compact('department', 'employees', 'filter', 'startDate', 'endDate'));
    }

    public function employeePerformance(Request $request, $employeeId)
    {
        $filter = $request->get('filter', 'month');
        $customStart = $request->get('custom_start');
        $customEnd = $request->get('custom_end');

        // Calculate date range
        $now = Carbon::now()->addHours(3);
        switch ($filter) {
            case 'day':
                $startDate = $now->copy()->startOfDay();
                $endDate = $now->copy()->endOfDay();
                break;
            case 'week':
                $startDate = $now->copy()->startOfWeek();
                $endDate = $now->copy()->endOfWeek();
                break;
            case 'month':
                $startDate = $now->copy()->startOfMonth();
                $endDate = $now->copy()->endOfMonth();
                break;
            case 'custom':
                $startDate = $customStart ? Carbon::parse($customStart) : $now->copy()->subMonth();
                $endDate = $customEnd ? Carbon::parse($customEnd) : $now->copy()->endOfDay();
                break;
            default:
                $startDate = $now->copy()->startOfMonth();
                $endDate = $now->copy()->endOfMonth();
        }

        $employee = \App\Models\Employee::with(['user', 'department'])->findOrFail($employeeId);

        $attendances = collect();
        $allAttendances = collect();

        if ($employee->user) {
            // Get performance records for the filtered date range
            $performanceLogs = \DB::table('performance_logs')
                ->where('employee_code', $employee->employee_code)
                ->whereBetween('check_time', [$startDate, $endDate])
                ->orderBy('check_time', 'desc')
                ->get();

            // Get attendance logs for check-in/out times
            $attendanceLogs = \DB::table('attendance_logs')
                ->where('employee_code', $employee->employee_code)
                ->whereBetween('check_time', [$startDate, $endDate])
                ->orderBy('check_time', 'asc')
                ->get();

            // Group attendance logs by date and merge with performance data
            $attendancesByDate = $attendanceLogs->groupBy(function($log) {
                return Carbon::parse($log->check_time)->format('Y-m-d');
            });

            // Merge attendance logs with performance logs
            $attendances = $performanceLogs->map(function($performance) use ($attendancesByDate) {
                $date = Carbon::parse($performance->check_time)->format('Y-m-d');
                $dayAttendance = $attendancesByDate->get($date, collect());

                $checkIn = $dayAttendance->firstWhere('type', 'check_in');
                $checkOut = $dayAttendance->firstWhere('type', 'check_out');

                $performance->check_in = $checkIn ? $checkIn->check_time : null;
                $performance->check_out = $checkOut ? $checkOut->check_time : null;

                return $performance;
            });

            // If no performance logs but we have attendance logs, create records from attendance
            if ($performanceLogs->isEmpty() && !$attendanceLogs->isEmpty()) {
                $attendances = $attendancesByDate->map(function($dayLogs, $date) {
                    $checkIn = $dayLogs->firstWhere('type', 'check_in');
                    $checkOut = $dayLogs->firstWhere('type', 'check_out');

                    $hoursWorked = 0;
                    if ($checkIn && $checkOut) {
                        $start = Carbon::parse($checkIn->check_time);
                        $end = Carbon::parse($checkOut->check_time);
                        $hoursWorked = $start->diffInHours($end);
                    }

                    return (object) [
                        'check_time' => $checkIn ? $checkIn->check_time : $date . ' 00:00:00',
                        'check_in' => $checkIn ? $checkIn->check_time : null,
                        'check_out' => $checkOut ? $checkOut->check_time : null,
                        'hours_worked' => $hoursWorked,
                        'status' => $checkIn ? 'present' : 'absent',
                        'notes' => null
                    ];
                })->values();
            }

            // Get ALL performance records for debugging
            $allAttendances = \DB::table('performance_logs')
                ->where('employee_code', $employee->employee_code)
                ->orderBy('check_time', 'desc')
                ->get();
        }
        $totalAttendances = $attendances->where('status', '!=', 'absent')->count();
        $expectedDays = $this->getExpectedWorkDays($startDate, $endDate);
        $attendanceRate = $expectedDays > 0 ? round(($totalAttendances / $expectedDays) * 100, 2) : 0;

        // Debug information
        $debugInfo = [
            'employee_code' => $employee->employee_code,
            'user_id' => $employee->user ? $employee->user->id : 'N/A',
            'filter' => $filter,
            'date_range' => $startDate->format('Y-m-d H:i') . ' to ' . $endDate->format('Y-m-d H:i'),
            'filtered_attendances' => $totalAttendances,
            'total_all_attendances' => $allAttendances->count(),
            'all_attendance_dates' => $allAttendances->take(5)->pluck('check_time')->map(function($date) {
                return Carbon::parse($date)->format('Y-m-d H:i');
            })->toArray()
        ];

        // Calculate average work hours from performance_logs
        $totalWorkHours = $attendances->sum('hours_worked');
        $daysWithHours = $attendances->where('hours_worked', '>', 0)->count();
        $averageWorkHours = $daysWithHours > 0 ? round($totalWorkHours / $daysWithHours, 2) : 0;

        return view('hr.attendance.employee-performance', compact(
            'employee', 'attendances', 'totalAttendances', 'expectedDays',
            'attendanceRate', 'averageWorkHours', 'filter', 'startDate', 'endDate', 'debugInfo'
        ));
    }

    private function getExpectedWorkDays($startDate, $endDate)
    {
        $start = Carbon::parse($startDate);
        $end = Carbon::parse($endDate);
        $workDays = 0;

        while ($start->lte($end)) {
            // Skip weekends (Saturday = 6, Sunday = 0)
            if ($start->dayOfWeek !== Carbon::FRIDAY && $start->dayOfWeek !== Carbon::SATURDAY) {
                $workDays++;
            }
            $start->addDay();
        }

        return $workDays;
    }

    public function createSampleData()
    {
        // Create sample attendance data for testing
        $users = User::whereHas('employee')->take(20)->get();
        $now = Carbon::now()->addHours(3);

        $created = 0;

        foreach ($users as $user) {
            // Create attendance for last 5 days
            for ($i = 0; $i < 5; $i++) {
                $date = $now->copy()->subDays($i);

                // Skip weekends
                if ($date->dayOfWeek === Carbon::FRIDAY || $date->dayOfWeek === Carbon::SATURDAY) {
                    continue;
                }

                // Random chance of attendance (80% chance)
                if (rand(1, 100) <= 80) {
                    $checkIn = $date->copy()->setTime(9, rand(0, 30), 0);
                    $checkOut = $checkIn->copy()->addHours(rand(7, 9))->addMinutes(rand(0, 59));

                    DB::table('_x_test_attendances')->insert([
                        'user_id' => $user->id,
                        'check_in' => $checkIn,
                        'check_out' => $checkOut,
                        'created_at' => $checkIn,
                        'updated_at' => $checkIn,
                    ]);

                    $created++;
                }
            }
        }

        return response()->json([
            'message' => "Created {$created} sample attendance records for testing",
            'redirect' => route('hr.attendance.dashboard')
        ]);
    }

    public function manualEntry()
    {
        $employees = \App\Models\Employee::with(['user', 'department'])
            ->whereHas('user')
            ->orderBy('first_name')
            ->get();

        $departments = \App\Models\Department::whereHas('employees')->orderBy('name')->get();

        return view('hr.attendance.manual-entry', compact('employees', 'departments'));
    }

    public function storeManual(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'date' => 'required|date',
            'check_in_time' => 'required',
            'check_out_time' => 'nullable',
        ]);

        $employee = \App\Models\Employee::with('user')->findOrFail($request->employee_id);

        if (!$employee->user) {
            return back()->with('error', 'Employee has no associated user account.');
        }

        $checkInDateTime = Carbon::parse($request->date . ' ' . $request->check_in_time)->addHours(3);
        $checkOutDateTime = null;

        if ($request->check_out_time) {
            $checkOutDateTime = Carbon::parse($request->date . ' ' . $request->check_out_time)->addHours(3);
        }

        // Check if attendance already exists for this date
        $existing = DB::table('_x_test_attendances')
            ->where('user_id', $employee->user->id)
            ->whereDate('check_in', $checkInDateTime->toDateString())
            ->first();

        if ($existing) {
            return back()->with('error', 'Attendance already exists for this employee on this date.');
        }

        DB::table('_x_test_attendances')->insert([
            'user_id' => $employee->user->id,
            'check_in' => $checkInDateTime,
            'check_out' => $checkOutDateTime,
            'created_at' => $checkInDateTime,
            'updated_at' => $checkInDateTime,
        ]);

        return back()->with('success', 'Attendance record created successfully.');
    }

    public function bulkEntry(Request $request)
    {
        $request->validate([
            'bulk_date' => 'required|date',
            'department_id' => 'required|exists:departments,id',
            'attendance_percentage' => 'required|integer|min:0|max:100',
        ]);

        $department = \App\Models\Department::with(['employees.user'])->findOrFail($request->department_id);
        $employees = $department->employees->filter(function($emp) { return $emp->user !== null; });

        $attendancePercentage = $request->attendance_percentage;
        $totalEmployees = $employees->count();
        $attendingEmployees = round(($attendancePercentage / 100) * $totalEmployees);

        $selectedEmployees = $employees->shuffle()->take($attendingEmployees);

        $created = 0;
        $date = Carbon::parse($request->bulk_date);

        foreach ($selectedEmployees as $employee) {
            // Check if attendance already exists
            $existing = DB::table('_x_test_attendances')
                ->where('user_id', $employee->user->id)
                ->whereDate('check_in', $date->toDateString())
                ->first();

            if (!$existing) {
                $checkIn = $date->copy()->setTime(9, rand(0, 30), 0)->addHours(3);
                $checkOut = $checkIn->copy()->addHours(rand(7, 9))->addMinutes(rand(0, 59));

                DB::table('_x_test_attendances')->insert([
                    'user_id' => $employee->user->id,
                    'check_in' => $checkIn,
                    'check_out' => $checkOut,
                    'created_at' => $checkIn,
                    'updated_at' => $checkIn,
                ]);

                $created++;
            }
        }

        return back()->with('success', "Created attendance for {$created} employees in {$department->name}.");
    }

    public function deleteAllAttendance()
    {
        $count = DB::table('_x_test_attendances')->count();
        DB::table('_x_test_attendances')->truncate();

        return response()->json([
            'message' => "Deleted {$count} attendance records",
            'redirect' => route('hr.attendance.dashboard')
        ]);
    }

    public function exportDepartmentAttendance(Request $request, $departmentId)
    {
        $filter = $request->get('filter', 'day');
        $customStart = $request->get('custom_start');
        $customEnd = $request->get('custom_end');

        // Calculate date range
        $now = Carbon::now()->addHours(3);
        switch ($filter) {
            case 'week':
                $startDate = $now->copy()->startOfWeek();
                $endDate = $now->copy()->endOfWeek();
                break;
            case 'month':
                $startDate = $now->copy()->startOfMonth();
                $endDate = $now->copy()->endOfMonth();
                break;
            case 'custom':
                $startDate = $customStart ? Carbon::parse($customStart) : $now->copy()->startOfDay();
                $endDate = $customEnd ? Carbon::parse($customEnd) : $now->copy()->endOfDay();
                break;
            default:
                $startDate = $now->copy()->startOfDay();
                $endDate = $now->copy()->endOfDay();
        }

        $department = \App\Models\Department::with(['employees.user'])->findOrFail($departmentId);

        // Prepare data for export
        $exportData = [];
        $exportData[] = [
            'Department',
            'Employee Name',
            'Employee Code',
            'Total Attendances',
            'Attendance Rate (%)',
            'Expected Work Days',
            'Status',
            'Period'
        ];

        $expectedDays = $this->getExpectedWorkDays($startDate, $endDate);

        foreach ($department->employees as $employee) {
            $attendanceCount = 0;
            $hasAttendance = false;

            if ($employee->user) {
                // Get performance logs for this employee
                $performanceRecords = \DB::table('performance_logs')
                    ->where('employee_code', $employee->employee_code)
                    ->whereBetween('check_time', [$startDate, $endDate])
                    ->where('status', '!=', 'absent')
                    ->get();

                $attendanceCount = $performanceRecords->count();
                $hasAttendance = $attendanceCount > 0;
            }

            $attendanceRate = $expectedDays > 0 ? round(($attendanceCount / $expectedDays) * 100, 2) : 0;
            $status = $hasAttendance ? 'Present' : 'Absent';

            $exportData[] = [
                $department->name,
                $employee->full_name ?? ($employee->user->name ?? 'Unknown'),
                $employee->employee_code ?? 'N/A',
                $attendanceCount,
                $attendanceRate,
                $expectedDays,
                $status,
                $startDate->format('Y-m-d') . ' to ' . $endDate->format('Y-m-d')
            ];
        }

        // Generate CSV
        $filename = 'department_attendance_' . str_replace(' ', '_', strtolower($department->name)) . '_' . $startDate->format('Y-m-d') . '_to_' . $endDate->format('Y-m-d') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($exportData) {
            $file = fopen('php://output', 'w');

            foreach ($exportData as $row) {
                fputcsv($file, $row);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function exportEmployeeAttendance(Request $request, $employeeId)
    {
        $filter = $request->get('filter', 'day');
        $customStart = $request->get('custom_start');
        $customEnd = $request->get('custom_end');

        // Calculate date range
        $now = Carbon::now()->addHours(3);
        switch ($filter) {
            case 'day':
                $startDate = $now->copy()->startOfDay();
                $endDate = $now->copy()->endOfDay();
                break;
            case 'week':
                $startDate = $now->copy()->startOfWeek();
                $endDate = $now->copy()->endOfWeek();
                break;
            case 'month':
                $startDate = $now->copy()->startOfMonth();
                $endDate = $now->copy()->endOfMonth();
                break;
            case 'custom':
                $startDate = $customStart ? Carbon::parse($customStart) : $now->copy()->startOfDay();
                $endDate = $customEnd ? Carbon::parse($customEnd) : $now->copy()->endOfDay();
                break;
            default:
                $startDate = $now->copy()->startOfDay();
                $endDate = $now->copy()->endOfDay();
        }

        $employee = \App\Models\Employee::with(['user', 'department'])->findOrFail($employeeId);

        // Prepare data for export
        $exportData = [];

        // Add header information
        $exportData[] = ['Employee Attendance Report'];
        $exportData[] = ['Employee Name:', $employee->full_name ?? ($employee->user->name ?? 'Unknown')];
        $exportData[] = ['Employee Code:', $employee->employee_code ?? 'N/A'];
        $exportData[] = ['Department:', $employee->department->name ?? 'N/A'];
        $exportData[] = ['Report Period:', $startDate->format('M d, Y') . ' - ' . $endDate->format('M d, Y')];
        $exportData[] = ['Generated On:', Carbon::now()->addHours(3)->format('M d, Y \a\t h:i A')];
        $exportData[] = []; // Empty row for spacing

        // Add column headers
        $exportData[] = [
            'Date',
            'Check In Time',
            'Check Out Time'
        ];

        $attendances = collect();

        if ($employee->user) {
            // Get performance records for the filtered date range
            $performanceLogs = \DB::table('performance_logs')
                ->where('employee_code', $employee->employee_code)
                ->whereBetween('check_time', [$startDate, $endDate])
                ->orderBy('check_time', 'desc')
                ->get();

            // Get attendance logs for check-in/out times
            $attendanceLogs = \DB::table('attendance_logs')
                ->where('employee_code', $employee->employee_code)
                ->whereBetween('check_time', [$startDate, $endDate])
                ->orderBy('check_time', 'asc')
                ->get();

            // Group attendance logs by date
            $attendancesByDate = $attendanceLogs->groupBy(function($log) {
                return Carbon::parse($log->check_time)->format('Y-m-d');
            });

            // If we have attendance logs, use them as the primary source
            if (!$attendanceLogs->isEmpty()) {
                $attendances = $attendancesByDate->map(function($dayLogs, $date) {
                    $checkIn = $dayLogs->firstWhere('type', 'check_in');
                    $checkOut = $dayLogs->firstWhere('type', 'check_out');

                    return (object) [
                        'check_time' => $checkIn ? $checkIn->check_time : ($checkOut ? $checkOut->check_time : $date . ' 00:00:00'),
                        'check_in' => $checkIn ? $checkIn->check_time : null,
                        'check_out' => $checkOut ? $checkOut->check_time : null,
                        'hours_worked' => 0,
                        'status' => $checkIn ? 'present' : 'absent',
                        'notes' => null
                    ];
                })->values();
            }
            // If no attendance logs but we have performance logs, use them
            elseif (!$performanceLogs->isEmpty()) {
                $attendances = $performanceLogs->map(function($performance) {
                    return (object) [
                        'check_time' => $performance->check_time,
                        'check_in' => null, // Performance logs don't have check-in/out times
                        'check_out' => null,
                        'hours_worked' => $performance->hours_worked ?? 0,
                        'status' => $performance->status ?? 'present',
                        'notes' => $performance->notes ?? null
                    ];
                });
            }
        }

        // Add attendance data to export
        foreach ($attendances as $attendance) {
            $checkInFormatted = $attendance->check_in ? Carbon::parse($attendance->check_in)->format('h:i A') : 'Not Checked In';
            $checkOutFormatted = $attendance->check_out ? Carbon::parse($attendance->check_out)->format('h:i A') : 'Not Checked Out';
            $date = Carbon::parse($attendance->check_time)->format('D, M d, Y');

            $exportData[] = [
                $date,
                $checkInFormatted,
                $checkOutFormatted
            ];
        }

        // If no attendance records, add a summary row
        if ($attendances->isEmpty()) {
            $exportData[] = [
                'No attendance records found for this period',
                'N/A',
                'N/A'
            ];
        }


        // Generate CSV filename
        $employeeName = str_replace(' ', '_', strtolower($employee->full_name ?? $employee->user->name ?? 'unknown'));
        $filename = 'employee_attendance_' . $employeeName . '_' . $startDate->format('Y-m-d') . '_to_' . $endDate->format('Y-m-d') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($exportData) {
            $file = fopen('php://output', 'w');

            foreach ($exportData as $row) {
                fputcsv($file, $row);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
