<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;

class AttendancePermit extends Model
{
    protected $table = 'attendance_permits';

    protected $fillable = [
        'employee_id',
        'permit_type_id',
        'requested_by_user_id',
        'leader_approved_by_user_id',
        'hr_approved_by_user_id',
        'start_datetime',
        'end_datetime',
        'all_day',
        'total_minutes',
        'member_reason',
        'leader_reason',
        'hr_reason',
        'status',
        'needs_hr',
        'leader_decision_at',
        'hr_decision_at',
        'cancelled_by_user_id',
        'cancelled_reason',
    ];

    protected $casts = [
        'start_datetime'      => 'datetime',
        'end_datetime'        => 'datetime',
        'leader_decision_at'  => 'datetime',
        'hr_decision_at'      => 'datetime',
        'all_day'             => 'boolean',
        'needs_hr'            => 'boolean',
    ];

    /* =======================
     |  Relationships
     ======================= */
    public function employee() { return $this->belongsTo(Employee::class); }
    public function type()     { return $this->belongsTo(AttendancePermitType::class, 'permit_type_id'); }

    public function requestedBy()      { return $this->belongsTo(User::class, 'requested_by_user_id'); }
    public function leaderApprovedBy() { return $this->belongsTo(User::class, 'leader_approved_by_user_id'); }
    public function hrApprovedBy()     { return $this->belongsTo(User::class, 'hr_approved_by_user_id'); }
    public function cancelledBy()      { return $this->belongsTo(User::class, 'cancelled_by_user_id'); }

    /* =======================
     |  Query Scopes
     ======================= */
    public function scopeForEmployee(Builder $q, int $employeeId): Builder {
        return $q->where('employee_id', $employeeId);
    }

    public function scopeStatusIn(Builder $q, array|string $status): Builder {
        return is_array($status) ? $q->whereIn('status', $status) : $q->where('status', $status);
    }

    public function scopeOverlapping(Builder $q, Carbon|string $from, Carbon|string $to): Builder {
        $from = $from instanceof Carbon ? $from : Carbon::parse($from);
        $to   = $to   instanceof Carbon ? $to   : Carbon::parse($to);

        // أي تداخل: يبدأ قبل نهاية الفترة وينتهي بعد بدايتها
        return $q->where(function ($qq) use ($from, $to) {
            $qq->where('start_datetime', '<=', $to)
               ->where('end_datetime', '>=', $from);
        });
    }

    /* =======================
     |  Helpers
     ======================= */
    public function durationMinutes(): int {
        if (!$this->start_datetime || !$this->end_datetime) return 0;
        return (int) $this->start_datetime->diffInMinutes($this->end_datetime, false);
    }

    public function isFinalApproved(): bool {
        return $this->status === 'hr_approved' || (!$this->needs_hr && $this->status === 'leader_approved');
    }

    public function isRejected(): bool {
        return $this->status === 'rejected';
    }
}
