<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute;

class AttendancePermitType extends Model
{
    protected $table = 'attendance_permit_types';

    protected $fillable = [
        'name',             // internal key: short_leave, early_exit, ...
        'display_name',     // UI label
        'requires_proof',   // needs attachment (e.g., medical)
        'count_as_work_time'
    ];

    protected $casts = [
        'requires_proof'    => 'boolean',
        'count_as_work_time'=> 'boolean',
    ];

    /* =======================
     |  Relationships
     ======================= */
    public function permits()
    {
        return $this->hasMany(AttendancePermit::class, 'permit_type_id');
    }

    /* =======================
     |  Accessors / Mutators
     ======================= */

    // اجبر name يبقى lower_snake خفيف عشان يفضل ثابت وفريد
    protected function name(): Attribute
    {
        return Attribute::make(
            set: fn ($value) => self::normalizeName($value),
        );
    }

    public static function normalizeName(?string $value): ?string
    {
        if ($value === null) return null;
        $v = trim($value);
        // حوّل مسافات وداش لأندر اسكور وخفض الحروف
        $v = preg_replace('/[\s\-]+/u', '_', $v);
        return strtolower($v);
    }

    /* =======================
     |  Scopes
     ======================= */
    public function scopeByName(Builder $q, string $name): Builder
    {
        return $q->where('name', self::normalizeName($name));
    }

    public function scopeRequiresProof(Builder $q): Builder
    {
        return $q->where('requires_proof', true);
    }

    public function scopeCountsAsWorkTime(Builder $q): Builder
    {
        return $q->where('count_as_work_time', true);
    }

    /* =======================
     |  Helpers
     ======================= */
    public function label(): string
    {
        return $this->display_name ?: $this->name;
    }
}
