<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Employee extends Model
{
    use HasFactory;

    protected $fillable = [
        'employee_code',
        'branch_id',
        'user_id',
        'department_id',
        'first_name',
        'last_name',
        'hire_date',
        'status'
    ];

    protected $casts = [
        'hire_date' => 'date',
    ];

    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function attendanceLogs()
    {
        // For composite key relationships, we need to use where clauses
        return $this->hasMany(AttendanceLog::class)
            ->where('employee_code', $this->employee_code)
            ->where('branch_id', $this->branch_id);
    }

    // Alternative method using a custom relationship
    public function attendanceLogsCustom()
    {
        return $this->hasMany(AttendanceLog::class, 'employee_code', 'employee_code')
            ->where('attendance_logs.branch_id', $this->branch_id);
    }

    // Another approach using a query builder
    public function getAttendanceLogsAttribute()
    {
        return AttendanceLog::where('employee_code', $this->employee_code)
            ->where('branch_id', $this->branch_id)
            ->get();
    }

    public function getFullNameAttribute()
    {
        return trim($this->first_name . ' ' . $this->last_name);
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    // Helper method to get attendance logs with proper composite key filtering
    public function getAttendanceLogs()
    {
        return AttendanceLog::where('employee_code', $this->employee_code)
            ->where('branch_id', $this->branch_id);
    }

    // jemy work start  from here  :
    public function department()
    {
        return $this->belongsTo(Department::class); // FK: department_id
    }
}
