<?php

namespace App\Notifications;

use App\Models\AttendancePermit;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\BroadcastMessage;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AttendancePermitStatusNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public AttendancePermit $permit,
        public string $action, // submitted|leader_approved|hr_approved|leader_rejected|hr_rejected|leader_approved_final|cancelled
        public ?User $actor = null
    ) {}

    public function via($notifiable): array
    {
        $channels = ['database', 'broadcast'];
        // if (!empty($notifiable->email)) $channels[] = 'mail';
        return $channels;
    }

    public function toMail($notifiable): MailMessage
    {
        $title = $this->title();
        return (new MailMessage)
            ->subject($title)
            ->greeting('مرحبًا،')
            ->line($this->body())
            ->action('عرض الإذن', url($this->openUrl())); // مطلق للإيميل
    }

    public function toArray($notifiable): array
    {
        return [
            'title'       => $this->title(),
            'body'        => $this->body(),
            'action'      => $this->action,
            'model'       => 'attendance_permit',
            'model_id'    => $this->permit->id,
            'status'      => (string)($this->permit->status ?? 'pending'),
            'open_url'    => $this->openUrl(), // ← path نسبي
            'actor'       => $this->actor?->only(['id', 'name', 'email']),
            'created_at'  => now()->toIso8601String(),
        ];
    }

    public function toBroadcast($notifiable): BroadcastMessage
    {
        return new BroadcastMessage($this->toArray($notifiable));
    }

    /* ===== Helpers ===== */

    protected function title(): string
    {
        $id = $this->permit->id;
        return match ($this->action) {
            'submitted'             => "طلب إذن جديد (#{$id})",
            'leader_approved'       => "تم اعتماد TL لإذن (#{$id})",
            'leader_approved_final' => "موافقة نهائية (بدون HR) لإذن (#{$id})",
            'hr_approved'           => "تم اعتماد HR لإذن (#{$id})",
            'leader_rejected'       => "تم رفض الإذن بواسطة TL (#{$id})",
            'hr_rejected'           => "تم رفض الإذن بواسطة HR (#{$id})",
            'cancelled'             => "تم إلغاء الإذن (#{$id})",
            default                 => "تحديث على إذن (#{$id})",
        };
    }

    protected function body(): string
    {
        $emp   = $this->permit->employee?->full_name
            ?: ($this->permit->employee?->user?->name ?? 'موظف');
        $type  = $this->permit->type?->display_name ?? $this->permit->type?->name ?? '—';
        $actor = $this->actor?->name ? "بواسطة {$this->actor->name}" : '';
        return "الموظف: {$emp} — النوع: {$type} — {$actor}";
    }

    protected function openUrl(): string
    {
        // route uses {id} as the parameter name (URI: dashboard/attendance/permits/{id})
        return route('permits.show.page', ['id' => $this->permit->id], false);
    }
}
