<?php

namespace App\Notifications;

use App\Models\Vacation;
use App\Models\User;
use App\Enums\VacationStatus;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\BroadcastMessage;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class VacationStatusNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Vacation $vacation,
        public string $action,   // submitted|leader_approved|hr_approved|leader_rejected|hr_rejected|cancelled ...
        public ?User $actor = null
    ) {}

    public function via($notifiable): array
    {
        $channels = ['database', 'broadcast'];
        // if (!empty($notifiable->email)) $channels[] = 'mail';
        return $channels;
    }

    public function toMail($notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject($this->title())
            ->greeting('مرحبًا،')
            ->line($this->body())
            ->action('عرض الطلب', url($this->openUrl())); // نحولها لمطلق في الإيميل
    }

    public function toArray($notifiable): array
    {
        return [
            'title'       => $this->title(),
            'body'        => $this->body(),
            'action'      => $this->action,
            'model'       => 'vacation',
            'model_id'    => $this->vacation->id,
            'status'      => $this->statusValue(),
            'open_url'    => $this->openUrl(), // ← path نسبي
            'actor'       => $this->actor?->only(['id', 'name', 'email']),
            'created_at'  => now()->toIso8601String(),
        ];
    }

    public function toBroadcast($notifiable): BroadcastMessage
    {
        return new BroadcastMessage($this->toArray($notifiable));
    }

    /* ===== Helpers ===== */

    protected function statusValue(): string
    {
        $s = $this->vacation->status;
        return $s instanceof VacationStatus ? $s->value : strtolower((string) $s);
    }

    protected function title(): string
    {
        $id = $this->vacation->id;
        return match ($this->action) {
            'submitted'        => "طلب إجازة جديد (#{$id})",
            'leader_approved'  => "تم اعتماد TL لطلب إجازة (#{$id})",
            'hr_approved'      => "تم اعتماد HR لطلب إجازة (#{$id})",
            'leader_rejected'  => "تم رفض الطلب بواسطة TL (#{$id})",
            'hr_rejected'      => "تم رفض الطلب بواسطة HR (#{$id})",
            'cancelled'        => "تم إلغاء طلب الإجازة (#{$id})",
            default            => "تحديث على طلب إجازة (#{$id})",
        };
    }

    protected function body(): string
    {
        $emp   = $this->vacation->employee?->full_name
            ?: ($this->vacation->employee?->user?->name ?? 'موظف');
        $type  = $this->vacation->type?->name ?? '—';
        $actor = $this->actor?->name ? "بواسطة {$this->actor->name}" : '';
        return "الموظف: {$emp} — النوع: {$type} — {$actor}";
    }

    protected function openUrl(): string
    {
        return route('vacations.show', ['vacation' => $this->vacation->id], false);
    }
}
