<?php

namespace App\Policies;

use App\Models\User;
use App\Models\AttendancePermit;

class AttendancePermitPolicy
{
    /* ===== helpers ===== */

    protected function can(User $u, string $ability): bool
    {
        return $u->can("permits.$ability");
    }

    protected function isAdminLike(User $u): bool
    {
        return $u->hasAnyRole(['Admin', 'Manager', 'HR']);
    }

    protected function isOwner(User $u, AttendancePermit $p): bool
    {
        if ($p->requested_by_user_id && $p->requested_by_user_id === $u->id) return true;
        $empUserId = optional($p->employee)->user_id;
        return $empUserId && $empUserId === $u->id;
    }

    protected function sameDepartmentByPermit(User $leader, AttendancePermit $p): bool
    {
        $leaderDeptId = optional($leader->employee)->department_id;
        $memberDeptId = optional($p->employee)->department_id;
        return $leaderDeptId && $memberDeptId && $leaderDeptId === $memberDeptId;
    }

    protected function isPending(AttendancePermit $p): bool
    {
        return strtolower((string)$p->status) === 'pending'
            || (isset($p->status) && strtolower((string)($p->status->value ?? $p->status)) === 'pending');
    }

    /* ===== abilities ===== */

    // دخول شاشة الفهرس
    public function viewAny(User $u): bool
    {
        if ($this->isAdminLike($u)) return true;
        if ($u->hasRole('Team Lead')) return true;

        // لو عنده أي من صلاحيات العرض المعتادة
        if ($this->can($u, 'viewAny') || $this->can($u, 'view') || $this->can($u, 'viewOwn')) {
            return true;
        }

        // سماح للـEmployee اللي عنده بطاقة موظف إنه يفتح الـIndex فاضي
        return (bool) optional($u->employee)->id;
    }

    // عرض سجل واحد
    public function view(User $u, AttendancePermit $p): bool
    {
        if ($this->isAdminLike($u)) return true;

        if (($this->can($u, 'viewOwn') || $this->can($u, 'view')) && $this->isOwner($u, $p)) {
            return true;
        }

        if ($u->hasRole('Team Lead')) {
            return $this->sameDepartmentByPermit($u, $p);
        }

        return false;
    }

    // إنشاء
    public function create(User $u): bool
    {
        return $this->can($u, 'create') || $this->isAdminLike($u) || $u->hasRole('Team Lead');
    }

    // تعديل — صاحب الطلب فقط أثناء Pending
    public function update(User $u, AttendancePermit $p): bool
    {
        return $this->isOwner($u, $p) && $this->isPending($p) && $this->can($u, 'update');
    }

    // إلغاء — صاحب الطلب فقط أثناء Pending (لا TL/HR/Admin)
    public function cancel(User $u, AttendancePermit $p): bool
    {
        return $this->isOwner($u, $p) && $this->isPending($p) && $this->can($u, 'cancel');
    }

    // موافقة قائد
    // public function leaderApprove(User $u, AttendancePermit $p): bool
    // {
    //     // if (!$this->isPending($p) || !$this->can($u, 'approve')) return false;
    //     // if (!$u->hasRole('Team Lead')) return false;
    //     // if ($this->isOwner($u, $p)) return false; // القائد لا يعتمد إذنه الشخصي
    //     // return $this->sameDepartmentByPermit($u, $p);
    //     // dd([
    //     //     'user_id' => $u->id,
    //     //     'employee_id' => $u->employee?->id,
    //     //     'permit_emp_id' => $p->employee_id,
    //     //     'permit_emp_dept' => $p->employee?->department_id,
    //     //     'user_emp_dept' => $u->employee?->department_id,
    //     // ]);


    //     if (!$u->hasRole('Team Lead')) return false;

    //     $employee = $u->employee;
    //     if (!$employee) return false;

    //     $permitEmp = $p->employee;
    //     if (!$permitEmp) return false;

    //     return $permitEmp->id === $employee->id
    //         || $permitEmp->department_id === $employee->department_id;
    // }

    public function leaderApprove(User $user, AttendancePermit $permit): bool
    {
        // mirror VacationPolicy: require pending + permission + role + not owner + same department
        if (!$this->isPending($permit) || !$this->can($user, 'approve')) return false;
        if (!$user->hasRole('Team Lead')) return false;
        if ($this->isOwner($user, $permit)) return false; // prevent TL approving own permit

        return $this->sameDepartmentByPermit($user, $permit);
    }

    /** هل HR أو Admin */
    protected function isHrOrAdmin(User $u): bool
    {
        return $u->hasRole('HR') || $u->hasRole('Admin');
    }


    // public function leaderApprove(User $user, AttendancePermit $permit): bool
    // {
    //     dd([
    //         'user_id' => $user->id,
    //         'employee_id' => $user->employee?->id,
    //         'permit_emp_id' => $permit->employee_id,
    //         'permit_emp_dept' => $permit->employee?->department_id,
    //         'user_emp_dept' => $user->employee?->department_id,
    //     ]);
    // }
    // موافقة HR/Admin (ومنع HR يعتمد إذنه الشخصي)
    /** موافقة HR (HR أو Admin فقط)، وممنوع اعتماد طلبه الشخصي */
    public function hrApprove(User $u, AttendancePermit $p): bool
    {
        if (!$this->isPending($p) || !$this->can($u, 'approve')) return false;
        if (!$this->isHrOrAdmin($u)) return false;

        // لو HR صاحب الطلب ممنوع يعتمد لنفسه
        if ($u->hasRole('HR') && $this->isOwner($u, $p)) return false;

        return true;
    }

    /** عام: هل يمكن اعتماد هذا الإذن (TL أو HR) */
    public function approve(User $u, AttendancePermit $p): bool
    {
        return $this->leaderApprove($u, $p) || $this->hrApprove($u, $p);
    }

    /** رفض — TL أو HR/Admin فقط (بنفس قواعد الإجازات) */
    public function reject(User $u, AttendancePermit $p): bool
    {
        if (!$this->isPending($p) || !$this->can($u, 'reject')) {
            return false;
        }

        // HR/Admin يقدروا يرفضوا
        if ($this->isHrOrAdmin($u)) {
            // لو HR هو صاحب الطلب، يمنع رفض/اعتماد طلبه الشخصي
            if ($u->hasRole('HR') && $this->isOwner($u, $p)) return false;
            return true;
        }

        // Team Lead يرفض نفس القسم، ولا يرفض طلبه الشخصي
        if ($u->hasRole('Team Lead')) {
            if ($this->isOwner($u, $p)) return false;
            return $this->sameDepartmentByPermit($u, $p);
        }

        return false;
    }
}
