<?php

namespace App\Policies;

use App\Models\User;
use App\Models\Vacation;
use App\Models\Employee;
use App\Enums\VacationStatus;

class VacationPolicy
{
    /** تحقق من صلاحية بصيغة vacations.{ability} */
    protected function can(User $u, string $ability): bool
    {
        return $u->can("vacations.$ability");
    }

    /** هل للمستخدم HR فقط أو Admin؟ (بدون Manager) */
    protected function isHrOrAdmin(User $u): bool
    {
        return $u->hasRole('HR') || $u->hasRole('Admin');
    }

    /** هل للمستخدم دور من الأدوار العليا القديمة؟ (لو حتحتاجها في view فقط) */
    protected function isAdminLikeView(User $u): bool
    {
        // للعرض فقط نسمح ل HR / Admin / Manager
        return $u->hasAnyRole(['Admin', 'Manager', 'HR']);
    }

    /** رجّع قيمة الـstatus كنص حتى لو كانت Enum */
    protected function statusStr(mixed $status): string
    {
        if ($status instanceof VacationStatus) {
            return strtolower($status->value ?? $status->name);
        }
        return strtolower((string) $status);
    }

    /** Pending ؟ */
    protected function isPending(\App\Models\Vacation $v): bool
    {
        return $v->status === \App\Enums\VacationStatus::PENDING
            || strtolower((string) ($v->status->value ?? $v->status)) === 'pending';
    }


    /** Team Leader & member في نفس القسم؟ (بالـuser_id) */
    protected function sameDepartment(User $leader, ?int $memberUserId): bool
    {
        if (!$memberUserId) return false;

        $leaderDeptId = optional($leader->employee)->department_id;
        if (!$leaderDeptId) return false;

        return Employee::where('user_id', $memberUserId)
            ->where('department_id', $leaderDeptId)
            ->exists();
    }

    /** ✔️ صاحب الطلب؟ */
    protected function isOwner(User $u, Vacation $v): bool
    {
        $employeeId = optional($u->employee)->id;
        return ($employeeId && $v->employee_id === $employeeId)
            || ($v->user_id && $v->user_id === $u->id);
    }

    /** ✔️ TL & vacation في نفس القسم */
    protected function sameDepartmentByVacation(User $leader, Vacation $v): bool
    {
        $leaderDeptId = optional($leader->employee)->department_id;
        $memberDeptId = optional($v->employee)->department_id;
        return $leaderDeptId && $memberDeptId && $leaderDeptId === $memberDeptId;
    }

    /**
     * عرض القائمة:
     * - Employee عنده viewOwn فقط: يدخل الصفحة (ويتفلتر في الكنترولر على سجلاته)
     * - TeamLeader/HR/Admin/Manager: يدخل الصفحة
     * - أو أي حد معاه vacations.viewAny / vacations.view / vacations.viewOwn
     */
    public function viewAny(User $u): bool
    {
        if ($this->isAdminLikeView($u)) return true;
        if ($u->hasRole('Team Lead')) return true;

        // لو عنده أي من صلاحيات العرض المعتادة
        if ($this->can($u, 'viewAny') || $this->can($u, 'view') || $this->can($u, 'viewOwn')) {
            return true;
        }

        // سماح للـEmployee اللي عنده بطاقة موظف إنه يفتح الـIndex فاضي
        return (bool) optional($u->employee)->id;
    }


    /**
     * عرض سجل معيّن:
     * 1) HR/Admin/Manager: مسموح
     * 2) صاحب الطلب (لو معاه viewOwn أو view)
     * 3) TeamLeader: لو نفس القسم
     */
    public function view(User $u, Vacation $v): bool
    {
        if ($this->isAdminLikeView($u)) return true;

        if (($this->can($u, 'viewOwn') || $this->can($u, 'view')) && $this->isOwner($u, $v)) {
            return true;
        }

        if ($u->hasRole('Team Lead')) {
            return $this->sameDepartmentByVacation($u, $v);
        }

        return false;
    }

    /**
     * إنشاء طلب: عادة كل Employee (و TL/HR/Admin ممكن)
     */
    public function create(User $u): bool
    {
        return $this->can($u, 'create') || $u->hasAnyRole(['Admin', 'HR', 'Team Lead']);
    }

    /**
     * تعديل:
     * - ✅ صاحب الطلب: لو Pending ومعاه vacations.update
     * - ❌ TL/HR/Admin/Manager: ممنوع التعديل طبقًا للقواعد الجديدة
     */
    public function update(User $u, Vacation $v): bool
    {
        return $this->isOwner($u, $v) && $this->isPending($v) && $this->can($u, 'update');
    }

    /**
     * حذف/إلغاء:
     * - ✅ صاحب الطلب فقط: لو Pending ومعاه vacations.delete
     * - ❌ لا TL ولا HR ولا Manager ولا Admin
     */
    public function delete(User $u, Vacation $v): bool
    {
        return $this->isOwner($u, $v) && $this->isPending($v) && $this->can($u, 'delete');
    }

    /**
     * موافقة عامة: TL أو HR (حسب أدوارهم)
     */
    public function approve(User $u, Vacation $v): bool
    {
        return $this->leaderApprove($u, $v) || $this->hrApprove($u, $v);
    }

    /**
     * رفض:
     * - ✅ TeamLeader: نفس القسم، وممنوع يرفض طلبه الشخصي
     * - ✅ HR/Admin: مسموح (كـ HR Reject)، بشرط Pending ومعاه vacations.reject
     * - ❌ Manager فقط بدون Admin/HR: غير مسموح هنا
     */
    public function reject(User $u, Vacation $v): bool
    {
        if (!$this->isPending($v) || !$this->can($u, 'reject')) {
            return false;
        }

        // HR/Admin يقدروا يرفضوا
        if ($this->isHrOrAdmin($u)) {
            // لو HR هو صاحب الطلب، يمنع رفض/اعتماد طلبه الشخصي
            if ($u->hasRole('HR') && $this->isOwner($u, $v)) return false;
            return true;
        }

        // Team Lead يرفض نفس القسم، ولا يرفض طلبه الشخصي
        if ($u->hasRole('Team Lead')) {
            if ($this->isOwner($u, $v)) return false;
            return $this->sameDepartmentByVacation($u, $v);
        }

        return false;
    }

    /**
     * موافقة Team Leader
     */
    public function leaderApprove(User $u, Vacation $v): bool
    {
        if (!$this->isPending($v) || !$this->can($u, 'approve')) return false;
        if (!$u->hasRole('Team Lead')) return false;
        if ($this->isOwner($u, $v)) return false; // منع اعتماد طلبه الشخصي

        return $this->sameDepartmentByVacation($u, $v);
    }

    /**
     * موافقة HR (HR أو Admin فقط)، وممنوع اعتماد طلبه الشخصي
     */
    public function hrApprove(User $u, Vacation $v): bool
    {
        if (!$this->isPending($v) || !$this->can($u, 'approve')) return false;
        if (!$this->isHrOrAdmin($u)) return false;

        // لو HR صاحب الطلب ممنوع يعتمد لنفسه
        if ($u->hasRole('HR') && $this->isOwner($u, $v)) return false;

        return true;
    }
}
