<?php

namespace App\Repositories;

use App\Models\User;
use App\Models\Attendance;
use App\Models\Department;
use Carbon\Carbon;

class ProfileRepository
{
    public function getUserAttendanceRecords($userId, $startDate, $endDate)
    {
        return Attendance::where('user_id', $userId)
            ->whereBetween('check_in', [$startDate, $endDate])
            ->orderBy('check_in')
            ->get();
    }

    public function getUserAttendanceCount($userId, $startDate)
    {
        return Attendance::where('user_id', $userId)
            ->where('check_in', '>=', $startDate)
            ->count();
    }

    public function getUserAttendanceTrend($userId, $currentMonth, $lastMonth)
    {
        $currentCount = Attendance::where('user_id', $userId)
            ->where('check_in', '>=', $currentMonth)
            ->count();

        $lastMonthCount = Attendance::where('user_id', $userId)
            ->whereBetween('check_in', [$lastMonth, $currentMonth])
            ->count();

        return [
            'current' => $currentCount,
            'previous' => $lastMonthCount,
            'trend' => $currentCount >= $lastMonthCount ? 'up' : 'down'
        ];
    }

    public function getDepartmentMemberCount($departmentId)
    {
        $department = Department::find($departmentId);
        return $department?->teamMembers()->count() ?? 0;
    }

    public function getUserProfile($userId)
    {
        return User::with(['employee', 'department'])
            ->find($userId);
    }

    public function getEmployeeAttendanceLogs($employee, $startDate, $endDate)
    {
        if (!$employee || !method_exists($employee, 'getAttendanceLogs')) {
            return collect();
        }

        return $employee->getAttendanceLogs()
            ->whereBetween('check_time', [$startDate, $endDate])
            ->orderBy('check_time')
            ->get();
    }

    public function getWorkingDaysCount($startDate, $endDate)
    {
        return $startDate->diffInWeekdays($endDate) + 1;
    }
}