<?php

namespace App\Services;

use Carbon\CarbonImmutable;
use Illuminate\Support\Facades\DB;

class AttendanceMergeService
{
    /**
     * يبني attendance_daily لفترة زمنية
     */
    public function buildDaily(string $fromDate, string $toDate): int
    {
        $from = CarbonImmutable::parse($fromDate)->startOfDay();
        $to   = CarbonImmutable::parse($toDate)->endOfDay();

        // ناخد كل الأحداث في الرينج
        $raw = DB::table('attendance_logs')
            ->selectRaw("
                employee_code,
                branch_id,
                DATE(CONVERT_TZ(check_time, @@session.time_zone, '+02:00')) as work_date,
                check_time,
                type,
                source
            ")
            ->whereBetween('check_time', [$from, $to]);

        // نجمع per day
        $rows = DB::query()->fromSub($raw, 'x')
            ->selectRaw("
                employee_code,
                MAX(branch_id) as branch_id,
                work_date,
                -- أول دخول من الجهاز (البصمة) ثم البيت
                MIN(CASE WHEN source='device' AND type='check_in'  THEN check_time END) as dev_first_in,
                MIN(CASE WHEN source='home'   AND type='check_in'  THEN check_time END) as home_first_in,
                -- آخر خروج من الجهاز ثم البيت
                MAX(CASE WHEN source='device' AND type='check_out' THEN check_time END) as dev_last_out,
                MAX(CASE WHEN source='home'   AND type='check_out' THEN check_time END) as home_last_out
            ")
            ->groupBy('employee_code', 'work_date')
            ->get();

        $upserts = 0;

        foreach ($rows as $r) {
            $firstIn = $r->dev_first_in ?? $r->home_first_in;
            $firstSrc = $r->dev_first_in ? 'device' : ($r->home_first_in ? 'home' : null);

            $lastOut = $r->dev_last_out ?? $r->home_last_out;
            $lastSrc = $r->dev_last_out ? 'device' : ($r->home_last_out ? 'home' : null);

            if (!$firstIn && !$lastOut) continue;

            // لو in من مصدر و out من مصدر آخر
            if ($firstSrc && $lastSrc && $firstSrc !== $lastSrc) {
                $firstSrc = 'merged';
                $lastSrc  = 'merged';
            }

            $data = [
                'employee_code'    => $r->employee_code,
                'work_date'        => $r->work_date,
                'first_in'         => $firstIn,
                'first_in_source'  => $firstSrc,
                'last_out'         => $lastOut,
                'last_out_source'  => $lastSrc,
                'branch_id'        => $r->branch_id,
                'updated_at'       => now(),
                'created_at'       => now(),
            ];

            DB::table('attendance_daily')->updateOrInsert(
                ['employee_code' => $r->employee_code, 'work_date' => $r->work_date],
                $data
            );
            $upserts++;
        }

        return $upserts;
    }

    /**
     * تحديث يوم واحد لموظف واحد (للاستخدام بعد كل حدث)
     */
    public function refreshDay(string $employeeCode, string $workDate): void
    {
        $this->buildDaily($workDate, $workDate);
    }
}
