<?php

namespace App\Services;

use App\Models\AttendancePermit;
use App\Models\CompanyHoliday;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Config;

class AttendancePermitRulesService
{
    public function blockWeekends(): bool
    {
        return (bool) Config::get('attendance.block_weekends', true);
    }

    public function blockHolidays(): bool
    {
        return (bool) Config::get('attendance.block_holidays', true);
    }

    public function weekendDays(): array
    {
        return array_map('strtolower', Config::get('attendance.weekend_days', ['fri','sat']));
    }

    public function isWeekend(Carbon $date): bool
    {
        $map = ['sun','mon','tue','wed','thu','fri','sat'];
        $dow = strtolower($map[$date->dayOfWeekIso % 7] ?? 'sun'); // dayOfWeekIso: 1=Mon..7=Sun
        // نعيد صياغتها لتوافق الخريطة:
        // تحويل ISO → اختصارنا: Mon..Sun → mon..sun
        $isoMap = [1=>'mon',2=>'tue',3=>'wed',4=>'thu',5=>'fri',6=>'sat',7=>'sun'];
        $dow = $isoMap[$date->dayOfWeekIso] ?? 'sun';
        return in_array($dow, $this->weekendDays(), true);
    }

    public function isHoliday(Carbon $date): bool
    {
        return CompanyHoliday::whereDate('date', $date->toDateString())->exists();
    }

    public function ensureNotBlockedInterval(Carbon $start, Carbon $end): void
    {
        // لو الإذن all-day أو فترة داخل يوم واحد: هنفحص اليومين (غالبًا نفس اليوم)
        // لو فترة تغطي أكثر من يوم، هنمشي عليهم كلهم ونمنع لو أي يوم بلوك (حسب سياستك)
        $cursor = $start->copy()->startOfDay();
        while ($cursor->lte($end)) {
            if ($this->blockWeekends() && $this->isWeekend($cursor)) {
                throw new \RuntimeException('لا يُسمح بالأذونات في عطلة نهاية الأسبوع.');
            }
            if ($this->blockHolidays() && $this->isHoliday($cursor)) {
                throw new \RuntimeException('لا يُسمح بالأذونات في العطلات الرسمية.');
            }
            $cursor->addDay();
        }
    }

    public function ensureMonthlyLimits(int $employeeId, ?string $permitTypeName, Carbon $start, Carbon $end): void
    {
        if (!$permitTypeName) return;

        $monthStart = $start->copy()->startOfMonth();
        $monthEnd   = $start->copy()->endOfMonth();

        // اجمع بيانات الشهر الجاري (موافق + معلّق؟ غالبًا نحسب المعتمَد فقط، أو الاتنين—اختر سياستك)
        $base = AttendancePermit::where('employee_id', $employeeId)
            ->whereBetween('start_datetime', [$monthStart, $monthEnd])
            ->whereIn('status', ['leader_approved','hr_approved','pending']); // اختَر حسب سياستك

        // فلترة بالاسم الداخلي لنوع الإذن
        $sumMinutes = (clone $base)
            ->whereHas('type', fn($q) => $q->where('name', $permitTypeName))
            ->sum('total_minutes');

        $count = (clone $base)
            ->whereHas('type', fn($q) => $q->where('name', $permitTypeName))
            ->count();

        $limitsMin  = config("attendance.monthly_limits.minutes.$permitTypeName");
        $limitsCnt  = config("attendance.monthly_limits.count.$permitTypeName");

        if ($limitsMin !== null) {
            $newMinutes = $start->diffInMinutes($end);
            if ($sumMinutes + $newMinutes > (int) $limitsMin) {
                throw new \RuntimeException('تم تجاوز الحد الشهري بالدقائق لنوع الإذن.');
            }
        }

        if ($limitsCnt !== null) {
            if ($count + 1 > (int) $limitsCnt) {
                throw new \RuntimeException('تم تجاوز عدد الطلبات الشهري لنوع الإذن.');
            }
        }
    }
}
