<?php

namespace App\Services;

use App\Models\AttendancePermit;
use App\Models\AttendancePermitType;
use App\Models\User;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use Illuminate\Validation\ValidationException;
use App\Services\AttendancePermitRulesService;
use App\Notifications\AttendancePermitStatusNotification;

class AttendancePermitService
{
    public function __construct(private AttendancePermitRulesService $rules) {}

    /** احسب إجمالي الدقائق */
    public function calculateTotalMinutes(Carbon $start, Carbon $end): int
    {
        return max(0, $start->diffInMinutes($end));
    }

    /** إنشاء إذن جديد */
    public function create(array $data, int $employeeId, int $requestedByUserId): AttendancePermit
    {
        return DB::transaction(function () use ($data, $employeeId, $requestedByUserId) {
            $type  = AttendancePermitType::findOrFail($data['permit_type_id']);
            $start = Carbon::parse($data['start_datetime']);
            $end   = Carbon::parse($data['end_datetime']);

            if ($end->lessThanOrEqualTo($start)) {
                abort(422, 'وقت النهاية يجب أن يكون بعد وقت البداية.');
            }

            // فحوص القواعد (ويك إند/عطلات + حدود شهرية)
            $this->rules->ensureNotBlockedInterval($start, $end);
            $this->rules->ensureMonthlyLimits($employeeId, $type->name ?? null, $start, $end);

            // إجمالي الدقائق
            $totalMinutes = $this->calculateTotalMinutes($start, $end);

            // احتساب الاحتياج لـ HR
            $needsHr = $this->computeNeedsHrOnCreate($type, $totalMinutes, $start, $end, $employeeId);

            $permit = AttendancePermit::create([
                'employee_id'                => $employeeId,
                'permit_type_id'             => $type->id,
                'start_datetime'             => $start,
                'end_datetime'               => $end,
                'total_minutes'              => $totalMinutes,
                'member_reason'              => $data['member_reason'] ?? null,
                'status'                     => 'pending',
                'needs_hr'                   => $needsHr,
                'requested_by_user_id'       => $requestedByUserId,
                'leader_reason'              => null,
                'hr_reason'                  => null,
                'leader_approved_by_user_id' => null,
                'hr_approved_by_user_id'     => null,
                'leader_decision_at'         => null,
                'hr_decision_at'             => null,
            ]);

            // إشعار بالإنشاء
            $this->notify($permit->fresh(), 'submitted', User::find($requestedByUserId));

            return $permit->refresh();
        });
        // (Notification for new request is handled by the controller to match Vacation flow)
    }

    public function update(AttendancePermit $permit, array $data): AttendancePermit
    {
        return DB::transaction(function () use ($permit, $data) {
            $type  = AttendancePermitType::findOrFail($data['permit_type_id']);
            $start = Carbon::parse($data['start_datetime']);
            $end   = Carbon::parse($data['end_datetime']);

            if ($end->lessThanOrEqualTo($start)) {
                abort(422, 'وقت النهاية يجب أن يكون بعد وقت البداية.');
            }

            // فحوص القواعد (ويك إند/عطلات + حدود شهرية)
            $this->rules->ensureNotBlockedInterval($start, $end);
            $this->rules->ensureMonthlyLimits($permit->employee_id, $type->name ?? null, $start, $end);

            $totalMinutes = $this->calculateTotalMinutes($start, $end);

            $needsHr = $this->computeNeedsHrOnCreate($type, $totalMinutes, $start, $end, $permit->employee_id);

            $permit->update([
                'permit_type_id' => $type->id,
                'start_datetime' => $start,
                'end_datetime'   => $end,
                'total_minutes'  => $totalMinutes,
                'member_reason'  => $data['member_reason'] ?? null,
                'needs_hr'       => $needsHr,
            ]);

            return $permit->refresh();
        });
    }

    /** اعتماد TL — فقط Pending */
    public function leaderApprove(AttendancePermit $permit, ?string $reason, int $approverUserId): AttendancePermit
    {
        return DB::transaction(function () use ($permit, $reason, $approverUserId) {
            if (strtolower((string) $permit->status) !== 'pending') {
                abort(422, 'لا يمكن اعتماد طلب غير Pending بواسطة التيم ليدر.');
            }

            $permit->update([
                'status'                      => 'leader_approved',
                'leader_reason'               => $reason,
                'leader_approved_by_user_id'  => $approverUserId,
                'leader_decision_at'          => now(),
            ]);

            $updatedPermit = AttendancePermit::find($permit->id);
            if (!$updatedPermit) {
                throw new \Exception('Permit not found after update.');
            }

            // Notify the permit owner (employee) about TL approval (service-level notification)
            if ($user = optional($updatedPermit->employee)->user) {
                $actor = User::find($approverUserId);
                // Use native notify() so each notifiable handles channels consistently
                $user->notify(new AttendancePermitStatusNotification($updatedPermit->fresh(), 'leader_approved', $actor));
            }

            return $updatedPermit;
        });
    }


    /** اعتماد HR — في pending أو leader_approved وقبل بداية الإذن */
    public function hrApprove(AttendancePermit $permit, ?string $reason, int $approverUserId): AttendancePermit
    {
        return DB::transaction(function () use ($permit, $reason, $approverUserId) {
            $status = strtolower((string) $permit->status);
            if (!in_array($status, ['pending', 'leader_approved'], true)) {
                abort(422, 'لا يمكن اعتماد هذا الطلب بواسطة HR في حالته الحالية.');
            }

            $this->assertHrWindowOpen($permit);

            $permit->update([
                'status'                   => 'hr_approved',
                'hr_reason'                => $reason,
                'hr_approved_by_user_id'   => $approverUserId,
                'hr_decision_at'           => now(),
                'needs_hr'                 => false,
            ]);

            // Notify the permit owner (employee) about HR approval
            if ($user = optional($permit->fresh()->employee)->user) {
                $actor = User::find($approverUserId);
                $user->notify(new AttendancePermitStatusNotification($permit->fresh(), 'hr_approved', $actor));
            }

            return $permit->refresh();
        });
    }

    /**
     * رفض (TL أو HR)
     * - TL: Pending فقط
     * - HR: Pending أو Leader_Approved + داخل نافذة HR
     */
    public function reject(AttendancePermit $permit, ?string $reason, string $by, int $approverUserId): AttendancePermit
    {
        return DB::transaction(function () use ($permit, $reason, $by, $approverUserId) {
            $actor = User::find($approverUserId);
            $by = strtolower($by);
            $status = strtolower((string) $permit->status);

            if ($by === 'hr') {
                if (!in_array($status, ['pending', 'leader_approved'], true)) {
                    abort(422, 'لا يمكن رفض هذا الطلب بواسطة HR في حالته الحالية.');
                }

                $this->assertHrWindowOpen($permit);

                $permit->update([
                    'status'                 => 'hr_rejected',
                    'hr_reason'              => $reason,
                    'hr_approved_by_user_id' => $approverUserId,
                    'hr_decision_at'         => now(),
                    'needs_hr'               => false,
                ]);

                // Notify the permit owner about HR rejection
                if ($user = optional($permit->fresh()->employee)->user) {
                    $user->notify(new AttendancePermitStatusNotification($permit->fresh(), 'hr_rejected', $actor));
                }
            } else {
                if ($status !== 'pending') {
                    throw ValidationException::withMessages(['status' => 'فقط الطلبات Pending يمكن رفضها بواسطة TL.']);
                }

                $permit->update([
                    'status'                      => 'leader_rejected',
                    'leader_reason'               => $reason,
                    'leader_approved_by_user_id'  => $approverUserId,
                    'leader_decision_at'          => now(),
                    'needs_hr'                    => false,
                ]);

                // Notify the permit owner about TL rejection
                if ($user = optional($permit->fresh()->employee)->user) {
                    $user->notify(new AttendancePermitStatusNotification($permit->fresh(), 'leader_rejected', $actor));
                }
            }

            return $permit->refresh();
        });
    }

    /** إلغاء منطقي بواسطة صاحب الطلب */
    public function cancelByMember(AttendancePermit $permit): AttendancePermit
    {
        $permit->update([
            'status' => 'cancelled_by_member',
        ]);
        // Controller will handle broadcasting and notifying recipients for cancellations
        return $permit->refresh();
    }

    /* ==================== Helpers ==================== */

    protected function computeNeedsHrOnCreate(
        AttendancePermitType $type,
        int $totalMinutes,
        Carbon $start,
        Carbon $end,
        int $employeeId
    ): bool {
        // 1) حد الدقائق
        $limitMinutes = (int) config('attendance.permits.needs_hr_if_minutes_over', 120);
        if ($totalMinutes > $limitMinutes) return true;

        // 2) خارج ساعات العمل؟
        if (config('attendance.permits.outside_working_hours_needs_hr', true)) {
            if ($this->isOutsideWorkingHours($start, $end)) return true;
        }

        // 3) تكرار شهري لنفس النوع؟
        $repeatThreshold = (int) config('attendance.permits.monthly_repeat_threshold', 3);
        if ($repeatThreshold > 0) {
            $countSameTypeThisMonth = AttendancePermit::where('employee_id', $employeeId)
                ->where('permit_type_id', $type->id)
                ->whereBetween('start_datetime', [now()->startOfMonth(), now()->endOfMonth()])
                ->count();
            if ($countSameTypeThisMonth >= $repeatThreshold) return true;
        }

        // 4) تعريف النوع نفسه (required_approvals === 2)
        if ((int) ($type->required_approvals ?? 1) === 2) return true;

        return false;
    }

    /** خارج ساعات العمل (بسيطة) */
    protected function isOutsideWorkingHours(Carbon $start, Carbon $end): bool
    {
        $workStart = data_get(config('attendance'), 'working_hours.start', '09:00');
        $workEnd   = data_get(config('attendance'), 'working_hours.end',   '18:00');

        [$wsH, $wsM] = array_map('intval', explode(':', $workStart));
        [$weH, $weM] = array_map('intval', explode(':', $workEnd));

        $startBound = $start->copy()->setTime($wsH, $wsM, 0);
        $endBound   = $start->copy()->setTime($weH, $weM, 0);

        return $start->lt($startBound) || $end->gt($endBound);
    }

    /** نافذة HR مفتوحة حتى بداية الإذن */
    protected function assertHrWindowOpen(AttendancePermit $p): void
    {
        $now = now();
        $deadline = $p->start_datetime instanceof Carbon
            ? $p->start_datetime->copy()
            : Carbon::parse($p->start_datetime);

        if ($now->greaterThanOrEqualTo($deadline->startOfMinute())) {
            abort(422, 'انتهت نافذة قرارات HR بعد بدء الإذن.');
        }
    }

    /** إشعارات موحّدة */
    protected function notify(AttendancePermit $permit, string $action, ?User $actor = null): void
    {
        // المرسل له: صاحب الطلب + TL + HR
        $targets = collect()
            ->when($permit->requestedBy, fn($c) => $c->push($permit->requestedBy))
            ->merge(User::role(['Team Lead', 'HR'])->get())
            ->unique('id')
            ->all();

        // Use native notify() for each target to ensure channels (database + broadcast) are applied per notifiable
        foreach ($targets as $t) {
            try {
                $t->notify(new AttendancePermitStatusNotification(
                    permit: $permit,
                    action: $action,
                    actor: $actor
                ));
            } catch (\Throwable $e) {
                // swallow to avoid breaking the flow; jobs will surface failures to the queue/failed_jobs
                report($e);
            }
        }
    }
}
