<?php

namespace App\Services;

use Carbon\Carbon;

class AttendanceService
{
    public function getAttendanceStats($user)
    {
        if (!$user->employee) {
            return $this->getEmptyAttendanceStats();
        }

        $employee = $user->employee;
        $currentMonth = Carbon::now()->startOfMonth();
        $currentMonthEnd = Carbon::now()->endOfMonth();

        $attendanceLogs = $employee->getAttendanceLogs()
            ->whereBetween('check_time', [$currentMonth, $currentMonthEnd])
            ->orderBy('check_time')
            ->get();

        // Group by date to get daily attendance
        $dailyAttendance = $attendanceLogs->groupBy(function($log) {
            return $log->check_time->format('Y-m-d');
        });

        $stats = [
            'present_days' => 0,
            'absent_days' => 0,
            'late_arrivals' => 0,
            'total_working_days' => 0,
            'attendance_rate' => 0
        ];

        // Calculate working days from hire date to current date
        $hireDate = Carbon::parse($employee->hire_date);
        $startDate = $hireDate->greaterThan($currentMonth) ? $hireDate : $currentMonth;
        $endDate = Carbon::now();

        // Count working days (excluding weekends)
        $workingDays = $startDate->diffInWeekdays($endDate) + 1;
        $stats['total_working_days'] = $workingDays;

        // Analyze daily attendance
        foreach ($dailyAttendance as $date => $logs) {
            $checkIns = $logs->where('type', 'check_in');
            $checkOuts = $logs->where('type', 'check_out');

            if ($checkIns->isNotEmpty()) {
                $stats['present_days']++;

                // Check if late (assuming 9:00 AM is standard time)
                $firstCheckIn = $checkIns->first();
                $checkInTime = $firstCheckIn->check_time->format('H:i:s');
                if ($checkInTime > '09:00:00') {
                    $stats['late_arrivals']++;
                }
            }
        }

        // Calculate absent days (working days - present days)
        $stats['absent_days'] = max(0, $workingDays - $stats['present_days']);

        // Calculate attendance rate and cap at 100%
        $stats['attendance_rate'] = $workingDays > 0
            ? min(100, round(($stats['present_days'] / $workingDays) * 100))
            : 0;

        return $stats;
    }

    public function getTodayAttendance($user)
    {
        if (!$user->employee) {
            return $this->getEmptyTodayAttendance();
        }

        $employee = $user->employee;
        $today = Carbon::today();
        $todayEnd = Carbon::today()->endOfDay();

        $todayLogs = $employee->getAttendanceLogs()
            ->whereBetween('check_time', [$today, $todayEnd])
            ->orderBy('check_time')
            ->get();

        $checkIns = $todayLogs->where('type', 'check_in');
        $checkOuts = $todayLogs->where('type', 'check_out');

        $attendance = [
            'has_checked_in' => $checkIns->isNotEmpty(),
            'has_checked_out' => $checkOuts->isNotEmpty(),
            'check_in_time' => null,
            'check_out_time' => null,
            'check_in_status' => 'absent',
            'check_out_status' => 'pending',
            'total_hours' => null
        ];

        if ($attendance['has_checked_in']) {
            $firstCheckIn = $checkIns->first();
            $attendance['check_in_time'] = $firstCheckIn->check_time->format('h:i A');

            // Determine check-in status (on time vs late)
            $checkInTime = $firstCheckIn->check_time->format('H:i:s');
            $attendance['check_in_status'] = $checkInTime <= '09:00:00' ? 'on_time' : 'late';
        }

        if ($attendance['has_checked_out']) {
            $lastCheckOut = $checkOuts->last();
            $attendance['check_out_time'] = $lastCheckOut->check_time->format('h:i A');
            $attendance['check_out_status'] = 'completed';

            // Calculate total hours if both check-in and check-out exist
            if ($attendance['has_checked_in']) {
                $checkInTime = $checkIns->first()->check_time;
                $checkOutTime = $lastCheckOut->check_time;
                $totalMinutes = $checkInTime->diffInMinutes($checkOutTime);
                $hours = floor($totalMinutes / 60);
                $minutes = $totalMinutes % 60;
                $attendance['total_hours'] = sprintf('%dh %dm', $hours, $minutes);
            }
        }

        return $attendance;
    }

    public function getRecentAttendanceRecords($user, $limit = 5)
    {
        if (!$user->employee) {
            return collect();
        }

        $employee = $user->employee;
        $startDate = Carbon::now()->subDays(30); // Last 30 days

        $attendanceLogs = $employee->getAttendanceLogs()
            ->where('check_time', '>=', $startDate)
            ->orderBy('check_time', 'desc')
            ->get()
            ->groupBy(function($log) {
                return $log->check_time->format('Y-m-d');
            });

        $records = [];
        $count = 0;

        foreach ($attendanceLogs as $date => $logs) {
            if ($count >= $limit) break;

            $checkIns = $logs->where('type', 'check_in');
            $checkOuts = $logs->where('type', 'check_out');

            $record = [
                'date' => Carbon::parse($date),
                'formatted_date' => Carbon::parse($date)->format('M d, Y'),
                'check_in_time' => null,
                'check_out_time' => null,
                'total_hours' => null,
                'status' => 'absent',
                'status_class' => 'bg-danger'
            ];

            if ($checkIns->isNotEmpty()) {
                $firstCheckIn = $checkIns->first();
                $record['check_in_time'] = $firstCheckIn->check_time->format('h:i A');

                // Determine status based on check-in time
                $checkInTime = $firstCheckIn->check_time->format('H:i:s');
                if ($checkInTime <= '09:00:00') {
                    $record['status'] = 'present';
                    $record['status_class'] = 'bg-success';
                } else {
                    $record['status'] = 'late';
                    $record['status_class'] = 'bg-warning';
                }

                if ($checkOuts->isNotEmpty()) {
                    $lastCheckOut = $checkOuts->last();
                    $record['check_out_time'] = $lastCheckOut->check_time->format('h:i A');

                    // Calculate total hours
                    $totalMinutes = $firstCheckIn->check_time->diffInMinutes($lastCheckOut->check_time);
                    $hours = floor($totalMinutes / 60);
                    $minutes = $totalMinutes % 60;
                    $record['total_hours'] = sprintf('%dh %dm', $hours, $minutes);
                } else {
                    // Check if it's today and still in progress
                    if (Carbon::parse($date)->isToday()) {
                        $record['status'] = 'in_progress';
                        $record['status_class'] = 'bg-info';
                    }
                }
            }

            $records[] = $record;
            $count++;
        }

        // Fill remaining slots with recent dates if needed
        if (count($records) < $limit) {
            $currentDate = Carbon::now();
            for ($i = 0; $i < $limit - count($records); $i++) {
                $date = $currentDate->copy()->subDays($i + count($records));

                // Skip if we already have this date
                $exists = collect($records)->contains(function($record) use ($date) {
                    return $record['date']->isSameDay($date);
                });

                if (!$exists && $date->isWeekday()) {
                    $records[] = [
                        'date' => $date,
                        'formatted_date' => $date->format('M d, Y'),
                        'check_in_time' => null,
                        'check_out_time' => null,
                        'total_hours' => null,
                        'status' => 'absent',
                        'status_class' => 'bg-danger'
                    ];
                }
            }
        }

        return collect($records)->sortByDesc('date')->take($limit);
    }

    private function getEmptyAttendanceStats()
    {
        return [
            'present_days' => 0,
            'absent_days' => 0,
            'late_arrivals' => 0,
            'total_working_days' => 0,
            'attendance_rate' => 0
        ];
    }

    private function getEmptyTodayAttendance()
    {
        return [
            'has_checked_in' => false,
            'has_checked_out' => false,
            'check_in_time' => null,
            'check_out_time' => null,
            'check_in_status' => 'absent',
            'check_out_status' => 'pending',
            'total_hours' => null
        ];
    }

    public function getTeamAttendance($user)
    {
        // Check if user has an employee record and belongs to a department
        if (!$user->employee || !$user->employee->department_id) {
            return collect();
        }

        $department = $user->employee->department;
        if (!$department) {
            return collect();
        }

        // Get all team members from the same department
        $teamMembers = $department->employees()
            ->where('id', '!=', $user->employee->id) // Exclude current user
            ->with('user')
            ->get();

        $today = Carbon::today();
        $todayEnd = Carbon::today()->endOfDay();
        $teamAttendance = [];

        foreach ($teamMembers as $employee) {
            // Get today's attendance logs for each team member
            $todayLogs = $employee->getAttendanceLogs()
                ->whereBetween('check_time', [$today, $todayEnd])
                ->orderBy('check_time')
                ->get();

            $checkIns = $todayLogs->where('type', 'check_in');
            $checkOuts = $todayLogs->where('type', 'check_out');

            $status = 'absent';
            $statusClass = 'absent';
            $statusLabel = 'Absent';
            $textClass = 'danger';

            if ($checkIns->isNotEmpty()) {
                $firstCheckIn = $checkIns->first();
                $checkInTime = $firstCheckIn->check_time->format('H:i:s');

                if ($checkInTime <= '09:00:00') {
                    $status = 'present';
                    $statusClass = 'present';
                    $statusLabel = 'Present';
                    $textClass = 'success';
                } else {
                    $status = 'late';
                    $statusClass = 'late';
                    $statusLabel = 'Late';
                    $textClass = 'warning';
                }
            }

            $teamAttendance[] = [
                'id' => $employee->id,
                'name' => $employee->user ? $employee->user->name : $employee->name,
                'email' => $employee->user ? $employee->user->email : $employee->email,
                'position' => $employee->position ?? 'Employee',
                'avatar' => $employee->user && $employee->user->avatar ? $employee->user->avatar : null,
                'status' => $status,
                'status_class' => $statusClass,
                'status_label' => $statusLabel,
                'text_class' => $textClass,
                'check_in_time' => $checkIns->isNotEmpty() ? $checkIns->first()->check_time->format('h:i A') : null,
                'check_out_time' => $checkOuts->isNotEmpty() ? $checkOuts->last()->check_time->format('h:i A') : null
            ];
        }

        return collect($teamAttendance);
    }

    public function getAllAttendanceRecords($user, $perPage = 15, $startDate = null, $endDate = null)
    {
        if (!$user->employee) {
            return collect();
        }

        $employee = $user->employee;

        $query = $employee->getAttendanceLogs()->orderBy('check_time', 'desc');

        // Apply date filtering if provided
        if ($startDate && $endDate) {
            $query->whereBetween('check_time', [
                Carbon::parse($startDate)->startOfDay(),
                Carbon::parse($endDate)->endOfDay()
            ]);
        }

        $attendanceLogs = $query->get()
            ->groupBy(function($log) {
                return $log->check_time->format('Y-m-d');
            });

        $records = [];

        foreach ($attendanceLogs as $date => $logs) {
            $checkIns = $logs->where('type', 'check_in');
            $checkOuts = $logs->where('type', 'check_out');

            $carbonDate = Carbon::parse($date);

            $record = [
                'date' => $carbonDate,
                'formatted_date' => $carbonDate->format('M d, Y'),
                'day_name' => $carbonDate->format('l'),
                'check_in_time' => null,
                'check_out_time' => null,
                'total_hours' => null,
                'status' => 'absent',
                'status_class' => 'bg-danger'
            ];

            if ($checkIns->isNotEmpty()) {
                $firstCheckIn = $checkIns->first();
                $record['check_in_time'] = $firstCheckIn->check_time->format('h:i A');

                $checkInTime = $firstCheckIn->check_time->format('H:i:s');
                if ($checkInTime <= '09:00:00') {
                    $record['status'] = 'present';
                    $record['status_class'] = 'bg-success';
                } else {
                    $record['status'] = 'late';
                    $record['status_class'] = 'bg-warning';
                }
            }

            if ($checkOuts->isNotEmpty()) {
                $lastCheckOut = $checkOuts->last();
                $record['check_out_time'] = $lastCheckOut->check_time->format('h:i A');

                if ($checkIns->isNotEmpty()) {
                    $checkInTime = $checkIns->first()->check_time;
                    $checkOutTime = $lastCheckOut->check_time;
                    $totalMinutes = $checkInTime->diffInMinutes($checkOutTime);
                    $hours = floor($totalMinutes / 60);
                    $minutes = $totalMinutes % 60;
                    $record['total_hours'] = sprintf('%dh %dm', $hours, $minutes);
                }
            } elseif ($checkIns->isNotEmpty()) {
                $record['status'] = 'in_progress';
                $record['status_class'] = 'bg-info';
            }

            $records[] = $record;
        }

        $collection = collect($records);

        // Calculate totals for all records
        $allTotals = [
            'total_present' => $collection->where('status', 'present')->count(),
            'total_late' => $collection->where('status', 'late')->count(),
            'total_absent' => $collection->where('status', 'absent')->count(),
            'total_in_progress' => $collection->where('status', 'in_progress')->count(),
            'total_records' => $collection->count()
        ];

        // Manual pagination
        $currentPage = request()->get('page', 1);
        $offset = ($currentPage - 1) * $perPage;
        $paginatedItems = $collection->slice($offset, $perPage);

        $paginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedItems,
            $collection->count(),
            $perPage,
            $currentPage,
            ['path' => request()->url(), 'query' => request()->query()]
        );

        // Add totals to paginator for use in view
        $paginator->allTotals = $allTotals;

        return $paginator;
    }
}