<?php
declare(strict_types=1);

namespace App\Services;

class DnsHealthService
{
    /** Normalize a host/domain (lowercase, trim dots/spaces) */
    protected function norm(?string $s): string {
        $s = trim((string)$s);
        $s = trim($s, ". \t\n\r\0\x0B");
        return strtolower($s);
    }

    /** Extract domain from a mail host like "mail.example.com" => "example.com" */
    public function baseDomainFromMailHost(?string $mailHost): ?string {
        $h = $this->norm($mailHost);
        if ($h === '') return null;
        // لو host يبدأ بـ mail. اشطبها
        if (str_starts_with($h, 'mail.') && substr_count($h, '.') >= 2) {
            return substr($h, 5);
        }
        // وإلا ارجع آخر 2-3 ليفلز كـ fallback
        $parts = explode('.', $h);
        if (count($parts) >= 2) {
            return implode('.', array_slice($parts, -2));
        }
        return $h;
    }

    /** DNS TXT records as strings */
    protected function txt(string $name): array {
        $name = $this->norm($name);
        if ($name === '') return [];
        $out = [];
        $records = @dns_get_record($name, DNS_TXT) ?: [];
        foreach ($records as $r) {
            if (!empty($r['txt'])) $out[] = (string)$r['txt'];
        }
        return $out;
    }

    /** A/AAAA records */
    protected function aaaa(string $name): array {
        $name = $this->norm($name);
        if ($name === '') return [];
        $A    = @dns_get_record($name, DNS_A)    ?: [];
        $AAAA = @dns_get_record($name, DNS_AAAA) ?: [];
        $ips  = [];
        foreach ($A as $r)    { if (!empty($r['ip']))      $ips[] = $r['ip']; }
        foreach ($AAAA as $r) { if (!empty($r['ipv6']))    $ips[] = $r['ipv6']; }
        return array_values(array_unique($ips));
    }

    /** MX records */
    protected function mx(string $domain): array {
        $domain = $this->norm($domain);
        if ($domain === '') return [];
        $mx = @dns_get_record($domain, DNS_MX) ?: [];
        $rows = [];
        foreach ($mx as $r) {
            $rows[] = [
                'host' => $this->norm($r['target'] ?? ''),
                'prio' => (int)($r['pri'] ?? $r['priority'] ?? 0),
            ];
        }
        usort($rows, fn($a,$b) => $a['prio'] <=> $b['prio']);
        return $rows;
    }

    /** PTR (reverse) for an IPv4 only (simple) */
    protected function ptr(?string $ip): ?string {
        if (!$ip) return null;
        $h = @gethostbyaddr($ip) ?: null;
        return $h ? $this->norm($h) : null;
    }

    /** Parse SPF from TXT */
    protected function spfInfo(string $domain): array {
        $txts = $this->txt($domain);
        $spf  = null;
        foreach ($txts as $t) {
            if (str_starts_with(strtolower($t), 'v=spf1')) { $spf = $t; break; }
        }
        if (!$spf) return ['ok'=>false, 'raw'=>null, 'tokens'=>[], 'policy'=>null];

        $tokens = preg_split('/\s+/', $spf);
        $policy = null;
        foreach (array_reverse($tokens) as $tok) {
            if (in_array($tok, ['-all','~all','?all','+all'], true)) { $policy = $tok; break; }
        }
        return ['ok'=>true, 'raw'=>$spf, 'tokens'=>$tokens, 'policy'=>$policy];
    }

    /** Check if SPF authorizes a given IP or mechanism (simple heuristic) */
    protected function spfAuthorizes(array $spf, ?string $ip, ?string $mailHost): bool {
    if (empty($spf['ok']) || !$ip) return false;
    $ip = trim($ip);
    $tokens = $spf['tokens'] ?? [];
    $tokens = array_map('strtolower', $tokens);

    // 1) ip4: match exact
    foreach ($tokens as $tok) {
        if (str_starts_with($tok, 'ip4:')) {
            $val = substr($tok, 4);
            // يدعم cidr كمان
            if ($val === $ip) return true;
            if (str_contains($val, '/')) {
                [$net, $cidr] = explode('/', $val, 2);
                if (filter_var($net, FILTER_VALIDATE_IP) && is_numeric($cidr)) {
                    $mask = ~((1 << (32 - (int)$cidr)) - 1);
                    $netLong = ip2long($net); $ipLong = ip2long($ip);
                    if (($netLong & $mask) === ($ipLong & $mask)) return true;
                }
            }
        }
    }

    // helpers to resolve A/AAAA
    $resolveA = function(string $name): array {
        $ips = [];
        foreach ((@dns_get_record($name, DNS_A) ?: []) as $r) if (!empty($r['ip'])) $ips[] = $r['ip'];
        return $ips;
    };

    // 2) a: domain A matches sender IP
    if (in_array('a', $tokens, true)) {
        foreach ($resolveA($this->norm($this->baseDomainFromMailHost($mailHost) ?? '')) as $aip) {
            if ($aip === $ip) return true;
        }
    }

    // 3) mx: any MX target A matches sender IP
    if (in_array('mx', $tokens, true)) {
        $domain = $this->baseDomainFromMailHost($mailHost) ?? '';
        foreach ((@dns_get_record($domain, DNS_MX) ?: []) as $mx) {
            $host = $this->norm($mx['target'] ?? '');
            foreach ($resolveA($host) as $aip) if ($aip === $ip) return true;
        }
    }

    // 4) include: اعتبره pass مبدئيًا (تحليل كامل يتطلب recursion)
    foreach ($tokens as $tok) if (str_starts_with($tok, 'include:')) return true;

    return false;
}


    /** DKIM selector info (default selector) */
    protected function dkimInfo(string $domain, string $selector = 'default'): array {
        $host = $selector.'._domainkey.'.$this->norm($domain);
        $txts = $this->txt($host);
        $raw  = null;
        foreach ($txts as $t) {
            if (str_contains(strtolower($t), 'v=dkim1') && str_contains($t, 'p=')) { $raw = $t; break; }
        }
        return [
            'ok'      => $raw !== null,
            'selector'=> $selector,
            'host'    => $host,
            'raw'     => $raw
        ];
    }

    /** DMARC info */
    protected function dmarcInfo(string $domain): array {
        $host = '_dmarc.'.$this->norm($domain);
        $txts = $this->txt($host);
        $raw  = null;
        foreach ($txts as $t) {
            if (str_starts_with(strtolower($t), 'v=dmarc1')) { $raw = $t; break; }
        }
        if (!$raw) return ['ok'=>false, 'host'=>$host, 'raw'=>null, 'policy'=>null, 'rua'=>null, 'adkim'=>null, 'aspf'=>null];

        $lower = strtolower($raw);
        $kv = [];
        foreach (explode(';', $lower) as $part) {
            $part = trim($part);
            if (!$part || !str_contains($part, '=')) continue;
            [$k,$v] = array_map('trim', explode('=', $part, 2));
            $kv[$k] = $v;
        }
        return [
            'ok'    => true,
            'host'  => $host,
            'raw'   => $raw,
            'policy'=> $kv['p'] ?? null,
            'rua'   => $kv['rua'] ?? null,
            'adkim' => $kv['adkim'] ?? null,
            'aspf'  => $kv['aspf'] ?? null,
        ];
    }

    /** Run all checks */
    public function run(?string $domain, ?string $mailHost): array {
        $mailHost = $this->norm($mailHost ?? '');
        $domain   = $this->norm($domain   ?? '') ?: $this->baseDomainFromMailHost($mailHost);

        // A/AAAA + PTR for mail host
        $ips = $mailHost ? $this->aaaa($mailHost) : [];
        $firstIpv4 = null;
        foreach ($ips as $ip) { if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) { $firstIpv4 = $ip; break; } }
        $ptr = $this->ptr($firstIpv4);

        // MX for domain
        $mx = $domain ? $this->mx($domain) : [];

        // SPF / DKIM / DMARC
        $spf   = $domain ? $this->spfInfo($domain) : ['ok'=>false];
        $spfOk = $this->spfAuthorizes($spf, $firstIpv4, $mailHost);
        $dkim  = $domain ? $this->dkimInfo($domain, 'default') : ['ok'=>false];
        $dmarc = $domain ? $this->dmarcInfo($domain) : ['ok'=>false];

        // Simple judgments
        $mxOk     = !empty($mx);
        $aOk      = !empty($ips);
        $ptrOk    = $ptr ? true : false;
        $dkimOk   = !empty($dkim['ok']);
        $spfExist = !empty($spf['ok']);
        $spfAuth  = $spfExist && $spfOk;
        $dmarcOk  = !empty($dmarc['ok']);

        return [
            'input'   => ['domain'=>$domain, 'mail_host'=>$mailHost],
            'a'       => ['ok'=>$aOk, 'ips'=>$ips],
            'ptr'     => ['ok'=>$ptrOk, 'value'=>$ptr, 'ip'=>$firstIpv4],
            'mx'      => ['ok'=>$mxOk, 'rows'=>$mx],
            'spf'     => ['exist'=>$spfExist, 'ok'=>$spfAuth, 'raw'=>$spf['raw'] ?? null, 'policy'=>$spf['policy'] ?? null],
            'dkim'    => $dkim,
            'dmarc'   => $dmarc,
            // quick tips
            'tips'    => [
                'alignment' => 'Ensure From/Envelope sender is @'.$domain.' and matches SMTP account.',
            ],
        ];
    }
}
