<?php
declare(strict_types=1);

namespace App\Services;

use App\Support\Sec;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Throwable;

// IMAP (Webklex)
use Webklex\IMAP\Facades\Client as Imap;   // Facade الصحيح
use Webklex\PHPIMAP\Client as ImapClient;  // الكائن الفعلي

// SMTP (Symfony Mailer)
use Symfony\Component\Mailer\Mailer;
use Symfony\Component\Mailer\Transport;
use Symfony\Component\Mime\Email;
use Symfony\Component\Mime\Address;

/**
 * Class EmailService
 *
 * Features:
 * - Dynamic per-user IMAP connection (creds من DB، مشفّرة).
 * - Lazy connect + error handling آمن.
 * - Inbox listing (light metadata) + pagination مبسّط.
 * - Fetch message by UID + attachments meta.
 * - Read/Unread flags (توافق مع إصدارات Webklex).
 * - List folders + basic search.
 * - Send via per-user SMTP (DSN سليم، Return-Path/EHLO، مرفقات).
 *
 * ENV (IMAP):
 *   IMAP_HOST, IMAP_PORT=993, IMAP_ENCRYPTION=ssl, IMAP_VALIDATE_CERT=true,
 *   IMAP_DEFAULT_FOLDER=INBOX, IMAP_TIMEOUT=30
 *
 * ENV (SMTP):
 *   MAIL_HOST, MAIL_PORT, MAIL_ENCRYPTION=ssl|tls|null,
 *   MAIL_EHLO_DOMAIN=mediafactoryworld.com (اختياري)،
 *   MAIL_VERIFY_PEER=true (اختياري), MAIL_TIMEOUT=30 (اختياري)
 *
 * Per-user columns (اقتراح):
 *   imap_username, imap_password_encrypted,
 *   smtp_username, smtp_password_encrypted, email_password (fallback)
 */
class EmailService
{
    /** @var ImapClient|null */
    private ?ImapClient $client = null;

    // ---------------------------------------------------------------------
    // IMAP: اتصال كسول + Utilities توافقية
    // ---------------------------------------------------------------------

    /**
     * أعِد عميل IMAP متصل (lazy-connect).
     *
     * @return ImapClient
     */
    protected function client(): ImapClient
    {
        if ($this->client instanceof ImapClient && $this->client->isConnected()) {
            return $this->client;
        }

        $u        = Auth::user();
        $username = $u->imap_username ?: $u->email;
        $password = Sec::dec($u->imap_password_encrypted);

        if (!$username || !$password) {
            abort(400, 'Mailbox is not connected for this user.');
        }

        $this->client = Imap::make([
            'host'          => env('IMAP_HOST'),
            'port'          => (int) env('IMAP_PORT', 993),
            'encryption'    => env('IMAP_ENCRYPTION', 'ssl'), // 'ssl' | 'tls' | null
            'validate_cert' => (bool) env('IMAP_VALIDATE_CERT', true),
            'username'      => $username,
            'password'      => $password,
            'protocol'      => 'imap',
            'timeout'       => (int) env('IMAP_TIMEOUT', 30),
        ]);

        try {
            $this->client->connect();
        } catch (Throwable $e) {
            report($e); // لا تطبع أسرار
            abort(502, 'Unable to connect to IMAP server.');
        }

        return $this->client;
    }

    /**
     * تحويل Attribute/Date مختلط إلى تاريخ مُنسّق.
     */
    private function fmtDate(mixed $attr, string $fmt = 'Y-m-d H:i'): ?string
    {
        if ($attr === null) return null;

        if ($attr instanceof \DateTimeInterface) {
            return $attr->format($fmt);
        }
        if (is_object($attr) && method_exists($attr, 'toDate')) {
            $dt = $attr->toDate();
            if ($dt instanceof \DateTimeInterface) return $dt->format($fmt);
        }
        if (is_object($attr) && method_exists($attr, 'get')) {
            $raw = $attr->get();
            if ($raw instanceof \DateTimeInterface) return $raw->format($fmt);
            if (is_string($raw) && $raw !== '') {
                try { return Carbon::parse($raw)->format($fmt); } catch (\Throwable) { return $raw; }
            }
        }
        if (is_string($attr) && $attr !== '') {
            try { return Carbon::parse($attr)->format($fmt); } catch (\Throwable) { return $attr; }
        }
        return null;
    }

    /**
     * Attribute → string بأمان (مع دعم Webklex Attribute).
     */
    private function attrToString(mixed $attr): ?string
    {
        if ($attr === null) return null;
        if (is_string($attr)) return $attr;
        if ($attr instanceof \Stringable) return (string) $attr;
        if (is_object($attr) && method_exists($attr, 'get')) {
            $v = $attr->get();
            if (is_string($v)) return $v;
            if ($v instanceof \Stringable) return (string) $v;
        }
        return null;
    }

    /**
     * توافق قراءة حالة Seen عبر إصدارات Webklex المختلفة.
     */
    private function isSeenCompat($msg): bool
    {
        try {
            if (method_exists($msg, 'isSeen')) {
                return (bool) $msg->isSeen();
            }
        } catch (\Throwable) {
            // ignore and try fallbacks
        }
        try {
            if (method_exists($msg, 'hasFlag')) {
                return (bool) ($msg->hasFlag('\Seen') || $msg->hasFlag('Seen') || $msg->hasFlag('SEEN'));
            }
        } catch (\Throwable) {}
        try {
            if (method_exists($msg, 'getFlags')) {
                $flags = $msg->getFlags();
                if (is_object($flags) && method_exists($flags, 'contains')) {
                    return $flags->contains('\Seen') || $flags->contains('Seen') || $flags->contains('SEEN');
                }
                if (is_object($flags) && method_exists($flags, 'has')) {
                    return $flags->has('\Seen') || $flags->has('Seen') || $flags->has('SEEN');
                }
                if (is_object($flags) && method_exists($flags, 'get')) {
                    $raw = $flags->get();
                    if (is_array($raw)) {
                        return in_array('\Seen', $raw, true) || in_array('Seen', $raw, true) || in_array('SEEN', $raw, true);
                    }
                }
                if (is_array($flags)) {
                    return in_array('\Seen', $flags, true) || in_array('Seen', $flags, true) || in_array('SEEN', $flags, true);
                }
            }
        } catch (\Throwable) {}
        return false;
    }

    /**
     * توافق التحقق من وجود مرفقات عبر الإصدارات.
     */
    private function hasAttachmentsCompat($msg): bool
    {
        try {
            if (method_exists($msg, 'hasAttachments')) {
                return (bool) $msg->hasAttachments();
            }
        } catch (\Throwable) {}
        try {
            if (method_exists($msg, 'getAttachments')) {
                $atts = $msg->getAttachments();
                if (is_iterable($atts)) {
                    foreach ($atts as $_) { return true; }
                }
            }
        } catch (\Throwable) {}
        return false;
    }

    // ---------------------------------------------------------------------
    // Inbox / Detail / Search
    // ---------------------------------------------------------------------

    /**
     * List inbox messages (lightweight metadata).
     *
     * @param  int         $page
     * @param  int         $limit
     * @param  string|null $folder
     * @return array{
     *   data: list<array{
     *     uid: mixed,
     *     subject: string,
     *     from: string,
     *     date: ?string,
     *     seen: bool,
     *     has_attachments: bool
     *   }>,
     *   pagination: array{page:int,limit:int,total:int,pages:int},
     *   folder: string
     * }
     */
    public function listInbox(int $page = 1, int $limit = 25, ?string $folder = null): array
    {
        $c          = $this->client();
        $folderName = $folder ?: env('IMAP_DEFAULT_FOLDER', 'INBOX');
        $f          = $c->getFolder($folderName);

        $collection = $f->messages()
            ->all()
            ->leaveUnread()
            ->since(Carbon::now()->subMonths(3))
            ->get()
            ->reverse(); // newest first

        $total  = $collection->count();
        $offset = max(0, ($page - 1) * $limit);

        $slice = $collection->slice($offset, $limit)->values();

        $data = $slice->map(function ($m) {
            return [
                'uid'             => $m->getUid(),
                'subject'         => $this->attrToString($m->getSubject()) ?? '(No subject)',
                'from'            => optional($m->getFrom()[0])->mail ?? '',
                'date'            => $this->fmtDate($m->getDate()),
                'seen'            => $this->isSeenCompat($m),
                'has_attachments' => $this->hasAttachmentsCompat($m),
            ];
        })->all();

        return [
            'data'       => $data,
            'pagination' => [
                'page'  => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => (int) ceil(max(1, $total) / max(1, $limit)),
            ],
            'folder'     => $folderName,
        ];
    }

    /**
     * Fetch one message by UID.
     *
     * @param  string      $uid
     * @param  string|null $folder
     * @return \Webklex\PHPIMAP\Message|\Webklex\PHPIMAP\Support\MessageCollection|mixed|null
     */
    public function getMessageByUID(string $uid, ?string $folder = null)
    {
        $c = $this->client();
        $f = $c->getFolder($folder ?: env('IMAP_DEFAULT_FOLDER', 'INBOX'));
        return $f->query()->uid($uid)->get()->first();
    }

    /**
     * Return attachment metadata list for a given UID (no streaming).
     *
     * @param  string      $uid
     * @param  string|null $folder
     * @return list<array{name:string,mime:?string,size:?int}>
     */
    public function getAttachmentsMeta(string $uid, ?string $folder = null): array
    {
        $msg = $this->getMessageByUID($uid, $folder);
        if (!$msg) return [];

        $atts = iterator_to_array($msg->getAttachments() ?? []);
        return array_values(array_map(function ($a) {
            return [
                'name' => $a->getName(),
                'mime' => $a->getMimeType(),
                'size' => $a->getSize(),
            ];
        }, $atts));
    }

    /**
     * Mark a message as read/unread (توافقية).
     */
    public function setSeen(string $uid, bool $seen = true, ?string $folder = null): bool
    {
        $msg = $this->getMessageByUID($uid, $folder);
        if (!$msg) return false;

        try {
            if ($seen) {
                if (method_exists($msg, 'setFlag')) {
                    return (bool) ($msg->setFlag('\Seen') || $msg->setFlag('Seen'));
                }
            } else {
                if (method_exists($msg, 'unsetFlag')) {
                    return (bool) ($msg->unsetFlag('\Seen') || $msg->unsetFlag('Seen'));
                }
            }
        } catch (\Throwable $e) {
            report($e);
        }
        return false;
    }

    /**
     * List folders.
     *
     * @return list<array{name:string,path:string}>
     */
    public function listFolders(): array
    {
        $c       = $this->client();
        $folders = $c->getFolders(); // array of Folder objects

        return collect($folders)->map(function ($f) {
            $name = $f->name ?? $f->path ?? ($f->full_name ?? 'INBOX');
            $path = $f->path ?? $f->full_name ?? $name;
            return ['name' => (string) $name, 'path' => (string) $path];
        })->all();
    }

    /**
     * Basic search (best-effort: q/from/since/before).
     *
     * @param  array<string,mixed> $params
     * @param  int                 $page
     * @param  int                 $limit
     * @param  string|null         $folder
     * @return array{
     *   data: list<array{uid:mixed,subject:string,from:string,date:?string,seen:bool,has_attachments:bool}>,
     *   pagination: array{page:int,limit:int,total:int,pages:int}
     * }
     */
    public function search(array $params = [], int $page = 1, int $limit = 25, ?string $folder = null): array
    {
        $c = $this->client();
        $f = $c->getFolder($folder ?: env('IMAP_DEFAULT_FOLDER', 'INBOX'));

        $q = $f->query()->leaveUnread();

        if (!empty($params['from']) && method_exists($q, 'from'))   { $q = $q->from($params['from']); }
        if (!empty($params['q'])    && method_exists($q, 'text'))   { $q = $q->text($params['q']); }
        if (!empty($params['since'])  && method_exists($q, 'since'))  { $q = $q->since(Carbon::parse($params['since'])); }
        if (!empty($params['before']) && method_exists($q, 'before')) { $q = $q->before(Carbon::parse($params['before'])); }

        $collection = $q->get()->reverse();

        $total  = $collection->count();
        $offset = max(0, ($page - 1) * $limit);
        $slice  = $collection->slice($offset, $limit)->values();

        $data = $slice->map(function ($m) {
            return [
                'uid'             => $m->getUid(),
                'subject'         => $this->attrToString($m->getSubject()) ?? '(No subject)',
                'from'            => optional($m->getFrom()[0])->mail ?? '',
                'date'            => $this->fmtDate($m->getDate()),
                'seen'            => $this->isSeenCompat($m),
                'has_attachments' => $this->hasAttachmentsCompat($m),
            ];
        })->all();

        return [
            'data'       => $data,
            'pagination' => [
                'page'  => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => (int) ceil(max(1, $total) / max(1, $limit)),
            ],
        ];
    }

    // ---------------------------------------------------------------------
    // SMTP: Send
    // ---------------------------------------------------------------------

    /**
     * Send an email via per-user SMTP using a robust DSN.
     *
     * DSN rules:
     *  - encryption=ssl  -> smtps://user:pass@host:465
     *  - encryption=tls  -> smtp://user:pass@host:587?encryption=tls
     *  - none            -> smtp://user:pass@host:25
     *
     * @param  string|array<int,string>                $to
     * @param  string                                  $subject
     * @param  string                                  $htmlBody
     * @param  array<int,string>|null                  $cc
     * @param  array<int,string>|null                  $bcc
     * @param  array<int,\Illuminate\Http\UploadedFile>|null $attachments
     * @return void
     */
    public function sendEmail(
        string|array $to,
        string $subject,
        string $htmlBody,
        ?array $cc = null,
        ?array $bcc = null,
        ?array $attachments = null
    ): void {
        $u = Auth::user();

        // ---- Resolve SMTP credentials (prefer dedicated SMTP cols) ----
        $smtpUser = $u->smtp_username ?? $u->imap_username ?? $u->email;
        $cipher   = $u->smtp_password_encrypted ?? $u->imap_password_encrypted ?? $u->email_password ?? null;
        $smtpPass = $cipher ? Crypt::decryptString($cipher) : null;

        if (!$smtpUser || !$smtpPass) {
            abort(400, 'SMTP credentials not configured for this user.');
        }

        // ---- Host/port/encryption (prefer MAIL_*, fallback IMAP_*) ----
        $host = (string) (env('MAIL_HOST') ?: env('IMAP_HOST', ''));
        if ($host === '') {
            abort(500, 'SMTP host is not configured (MAIL_HOST or IMAP_HOST).');
        }

        $enc  = strtolower((string) (env('MAIL_ENCRYPTION') ?: env('IMAP_ENCRYPTION', 'tls'))); // ssl|tls|''.
        $enc  = in_array($enc, ['ssl','tls'], true) ? $enc : '';
        $port = (int) (env('MAIL_PORT') ?: ($enc === 'ssl' ? 465 : ($enc === 'tls' ? 587 : 25)));

        $scheme = $enc === 'ssl' ? 'smtps' : 'smtp';

        // DSN options
        $qs = [];

        // STARTTLS
        if ($scheme === 'smtp' && $enc === 'tls') {
            $qs['encryption'] = 'tls';
        }

        // EHLO domain (Optional but recommended)
        $ehlo = (string) env('MAIL_EHLO_DOMAIN', parse_url(config('app.url'), PHP_URL_HOST) ?: 'localhost');
        if ($ehlo !== '') {
            $qs['local_domain'] = $ehlo;
        }

        // Timeouts (seconds)
        $timeout = (int) env('MAIL_TIMEOUT', 30);
        if ($timeout > 0) {
            $qs['timeout'] = (string) $timeout;
        }

        // Peer verification (⚠️ يفضل true في الإنتاج)
        $verifyPeer = (bool) (env('MAIL_VERIFY_PEER', env('IMAP_VALIDATE_CERT', true)));
        if ($verifyPeer === false) {
            $qs['verify_peer']       = '0';
            $qs['verify_peer_name']  = '0';
            $qs['allow_self_signed'] = '1';
        }

        $query = $qs ? ('?' . http_build_query($qs)) : '';

        $dsn = sprintf(
            '%s://%s:%s@%s:%d%s',
            $scheme,
            rawurlencode($smtpUser),
            rawurlencode($smtpPass),
            $host,
            $port,
            $query
        );

        // ---- Build transport ----
        try {
            $transport = Transport::fromDsn($dsn);
        } catch (\InvalidArgumentException $e) {
            report($e);
            abort(500, 'Invalid SMTP DSN. Check host/port/encryption.');
        }

        $mailer = new Mailer($transport);

        // ---- Build message ----
        // بعض الخوادم تُلزم أن يكون From مطابقًا لحساب SMTP
        $fromAddress = $u->email ?: $smtpUser;
        if (!str_contains((string) $fromAddress, '@')) {
            $fromAddress = $smtpUser;
        }

        // نص عادي بديل (يحسن القابلية للتسليم)
        $plainText = trim(preg_replace('/\s+/', ' ', html_entity_decode(strip_tags($htmlBody))));

        $email = (new Email())
            ->from(new Address($fromAddress, $u->name ?? null))
            ->to(...(array) $to)
            ->subject($subject)
            ->text($plainText !== '' ? $plainText : $subject)
            ->html($htmlBody)
            // Return-Path/Envelope sender لِـ DMARC/SPF alignment
            ->returnPath(new Address($smtpUser));

        // Debug header لتتبع الرسائل في Track Delivery
        $email->getHeaders()->addTextHeader('X-App-Mail-Debug', 'app=yg; user='.(string) $u->id.'; ts='.now()->toIso8601String());

        if ($cc)  { $email->cc(...$cc); }
        if ($bcc) { $email->bcc(...$bcc); }

        if ($attachments) {
            foreach ($attachments as $file) {
                if ($file instanceof \Illuminate\Http\UploadedFile && $file->isValid()) {
                    $email->attachFromPath(
                        $file->getRealPath(),
                        $file->getClientOriginalName(),
                        $file->getMimeType()
                    );
                }
            }
        }

        // ---- Send ----
        try {
            $mailer->send($email);
        } catch (Throwable $e) {
            report($e);
            abort(502, 'SMTP send failed.');
        }
    }
}
