<?php
declare(strict_types=1);

namespace App\Services;

use App\Support\Sec;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Webklex\PHPIMAP\ClientManager;
use Symfony\Component\Mailer\Transport;

class MailDiagnostics
{
    /** Get SMTP config (config() first, then env()) */
    protected function smtpConfig(): array {
        $cfg = (array) config('mail.mailers.smtp', []);

        $host = env('MAIL_HOST', $cfg['host'] ?? '');
        $enc  = strtolower((string) env('MAIL_ENCRYPTION', $cfg['encryption'] ?? 'tls')); // ssl|tls|''
        $port = (int) env('MAIL_PORT', $cfg['port'] ?? 0);
        if (!$port) $port = $enc === 'ssl' ? 465 : ($enc === 'tls' ? 587 : 25);

        $verify = env('MAIL_VERIFY_PEER', null);
        if ($verify === null) {
            // fallback إلى IMAP_VALIDATE_CERT لو متوفر
            $verify = env('IMAP_VALIDATE_CERT', true);
        }

        return [
            'host' => $host,
            'port' => $port,
            'enc'  => in_array($enc, ['ssl','tls'], true) ? $enc : '',
            'verify_peer' => (bool) $verify,
        ];
    }

    /** Get IMAP config (Webklex) from config(), then env() */
    protected function imapConfig(): array {
        $defaultAccount = config('imap.default', 'default');
        $a = (array) config("imap.accounts.$defaultAccount", []);

        $host = $a['host'] ?? env('IMAP_HOST', '');
        $port = (int) ($a['port'] ?? env('IMAP_PORT', 993));
        $enc  = $a['encryption'] ?? env('IMAP_ENCRYPTION', 'ssl');
        $validate = (bool) ($a['validate_cert'] ?? env('IMAP_VALIDATE_CERT', true));

        return [
            'host' => $host,
            'port' => $port,
            'enc'  => $enc,
            'validate' => $validate,
            'folder' => config('imap.options.common.folder', env('IMAP_DEFAULT_FOLDER', 'INBOX')),
            'timeout'=> (int) env('IMAP_TIMEOUT', 30),
        ];
    }

    /** Build SMTP DSN using config + per-user creds (same rules as EmailService) */
    public function buildSmtpDsn(object $user): array {
        $smtp = $this->smtpConfig();

        $smtpUser = $user->smtp_username ?? $user->imap_username ?? $user->email;
        $cipher   = $user->smtp_password_encrypted ?? $user->imap_password_encrypted ?? $user->email_password ?? null;
        $smtpPass = $cipher ? Crypt::decryptString($cipher) : null;

        $scheme = $smtp['enc'] === 'ssl' ? 'smtps' : 'smtp';
        $qs = [];
        if ($scheme === 'smtp' && $smtp['enc'] === 'tls') {
            $qs['encryption'] = 'tls'; // STARTTLS
        }
        if ($smtp['verify_peer'] === false) {
            $qs['verify_peer']       = '0';
            $qs['verify_peer_name']  = '0';
            $qs['allow_self_signed'] = '1';
        }
        $query = $qs ? ('?' . http_build_query($qs)) : '';

        $dsn = sprintf(
            '%s://%s:%s@%s:%d%s',
            $scheme,
            rawurlencode((string) $smtpUser),
            rawurlencode((string) $smtpPass),
            $smtp['host'],
            $smtp['port'],
            $query
        );

        $dsnSanitized = sprintf(
            '%s://%s:%s@%s:%d%s',
            $scheme,
            (string) $smtpUser,
            '***',
            $smtp['host'],
            $smtp['port'],
            $query
        );

        return [$dsn, $dsnSanitized, $smtp];
    }

    /** DNS resolve helper */
    protected function dnsResolve(?string $host): array {
        if (!$host) return ['ok'=>false,'host'=>null,'ip'=>null];
        $ip = gethostbyname($host);
        $ok = filter_var($ip, FILTER_VALIDATE_IP) !== false;
        return ['ok' => $ok, 'host' => $host, 'ip' => $ok ? $ip : null];
        }

    /** TCP check helper */
    protected function tcpCheck(?string $host, int $port, int $timeout = 5): array {
        if (!$host) return ['ok'=>false,'error'=>'no host'];
        $errno = 0; $errstr = '';
        $start = microtime(true);
        $fp = @fsockopen($host, $port, $errno, $errstr, $timeout);
        $dt = round((microtime(true) - $start) * 1000);
        if ($fp) { fclose($fp); return ['ok'=>true,'ms'=>$dt]; }
        return ['ok'=>false,'ms'=>$dt,'error'=>trim($errstr ?: ('errno '.$errno))];
    }

    /** Run all checks */
    public function run(): array {
        $u = Auth::user();
        $smtp = $this->smtpConfig();
        $imap = $this->imapConfig();

        // ENV snapshot (فعليًا بنعرض القيم الفعّالة بعد fallback)
        $env = [
            'MAIL_HOST'         => $smtp['host'] ?: '—',
            'MAIL_PORT'         => $smtp['port'] ?: '—',
            'MAIL_ENCRYPTION'   => $smtp['enc'] ?: '—',
            'MAIL_VERIFY_PEER'  => $smtp['verify_peer'],
            'IMAP_HOST'         => $imap['host'] ?: '—',
            'IMAP_PORT'         => $imap['port'] ?: '—',
            'IMAP_ENCRYPTION'   => $imap['enc'] ?: '—',
            'IMAP_VALIDATE_CERT'=> $imap['validate'],
            'IMAP_DEFAULT_FOLDER'=> $imap['folder'] ?: 'INBOX',
        ];

        // DNS
        $dns = [
            'smtp' => $this->dnsResolve($smtp['host']),
            'imap' => $this->dnsResolve($imap['host']),
        ];

        // TCP
        $tcp = [
            'smtp' => $this->tcpCheck($smtp['host'], (int) $smtp['port']),
            'imap' => $this->tcpCheck($imap['host'], (int) $imap['port']),
        ];

        // IMAP login test
        $imapLogin = ['ok'=>false,'error'=>null,'folders'=>0];
        try {
            $username = $u->imap_username ?? $u->email;
            $password = $u->imap_password_encrypted ? Sec::dec($u->imap_password_encrypted) : null;
            if (!$username || !$password) {
                throw new \RuntimeException('Missing per-user IMAP credentials.');
            }

            $cm = new ClientManager();
            $client = $cm->make([
                'host'          => $imap['host'],
                'port'          => $imap['port'],
                'encryption'    => $imap['enc'],
                'validate_cert' => $imap['validate'],
                'username'      => $username,
                'password'      => $password,
                'protocol'      => 'imap',
                'timeout'       => $imap['timeout'],
            ]);
            $client->connect();
            $folders = $client->getFolders();
            $imapLogin['ok'] = true;
            $imapLogin['folders'] = is_array($folders) ? count($folders) : (method_exists($folders,'count') ? $folders->count() : 0);
        } catch (\Throwable $e) {
            $imapLogin['ok'] = false;
            $imapLogin['error'] = $e->getMessage();
        }

        // SMTP transport start()
        [$dsn, $dsnSan, $smtpEff] = $this->buildSmtpDsn($u);
        $smtpStart = ['ok'=>false, 'error'=>null, 'dsn'=>$dsnSan];
        try {
            $transport = Transport::fromDsn($dsn);
            if (method_exists($transport, 'start')) $transport->start();
            $smtpStart['ok'] = true;
        } catch (\Throwable $e) {
            $smtpStart['ok'] = false;
            $smtpStart['error'] = $e->getMessage();
        }

        return [
            'env'  => $env,
            'dns'  => $dns,
            'tcp'  => $tcp,
            'imap' => $imapLogin,
            'smtp' => $smtpStart,
        ];
    }
}
