<?php

namespace App\Services;

use Carbon\Carbon;

class PerformanceService
{
    public function getPerformanceData($user, $type = 'week', $startDate = null, $endDate = null)
    {
        if ($type === 'custom' && $startDate && $endDate) {
            $start = Carbon::parse($startDate);
            $end = Carbon::parse($endDate);
        } elseif ($type === 'month') {
            $start = Carbon::now()->startOfMonth();
            $end = Carbon::now()->endOfMonth();
        } else {
            // Default to week
            $start = Carbon::now()->startOfWeek();
            $end = Carbon::now()->endOfWeek();
        }

        return $this->calculatePerformanceMetrics($user, $start, $end, $type);
    }

    public function getInitialPerformanceData($user, $type = 'week')
    {
        if ($type === 'month') {
            $start = Carbon::now()->startOfMonth();
            $end = Carbon::now()->endOfMonth();
        } else {
            // Default to week
            $start = Carbon::now()->startOfWeek();
            $end = Carbon::now()->endOfWeek();
        }

        return $this->calculatePerformanceMetrics($user, $start, $end, $type);
    }

    private function calculatePerformanceMetrics($user, $startDate, $endDate, $type = 'week')
    {
        if (!$user->employee) {
            return $this->getEmptyPerformanceData($type);
        }

        $employee = $user->employee;

        // Get attendance logs for the period
        $attendanceLogs = $employee->getAttendanceLogs()
            ->whereBetween('check_time', [$startDate, $endDate])
            ->orderBy('check_time')
            ->get();

        // Group by date
        $dailyAttendance = $attendanceLogs->groupBy(function($log) {
            return $log->check_time->format('Y-m-d');
        });

        // Generate date range labels
        $labels = [];
        $attendanceData = [];
        $hoursWorkedData = [];
        $statusData = ['present' => 0, 'late' => 0, 'absent' => 0];

        $currentDate = $startDate->copy();
        while ($currentDate <= $endDate) {
            if ($currentDate->isWeekday()) { // Only count working days
                $dateStr = $currentDate->format('Y-m-d');
                $labels[] = $type === 'week'
                    ? $currentDate->format('D')
                    : $currentDate->format('M j');

                if (isset($dailyAttendance[$dateStr])) {
                    $dayLogs = $dailyAttendance[$dateStr];
                    $checkIns = $dayLogs->where('type', 'check_in');
                    $checkOuts = $dayLogs->where('type', 'check_out');

                    if ($checkIns->isNotEmpty()) {
                        $firstCheckIn = $checkIns->first();
                        $checkInTime = $firstCheckIn->check_time->format('H:i:s');

                        // Determine status
                        if ($checkInTime <= '09:00:00') {
                            $attendanceData[] = 1; // Present
                            $statusData['present']++;
                        } else {
                            $attendanceData[] = 0.5; // Late
                            $statusData['late']++;
                        }

                        // Calculate hours worked
                        if ($checkOuts->isNotEmpty()) {
                            $lastCheckOut = $checkOuts->last();
                            $hoursWorked = $firstCheckIn->check_time->diffInHours($lastCheckOut->check_time);
                            $hoursWorkedData[] = round($hoursWorked, 2);
                        } else {
                            // If no checkout, calculate partial hours or set to 0
                            $hoursWorkedData[] = $currentDate->isToday()
                                ? $firstCheckIn->check_time->diffInHours(Carbon::now())
                                : 0;
                        }
                    } else {
                        $attendanceData[] = 0; // Absent
                        $hoursWorkedData[] = 0;
                        $statusData['absent']++;
                    }
                } else {
                    $attendanceData[] = 0; // Absent
                    $hoursWorkedData[] = 0;
                    if (!$currentDate->isFuture()) {
                        $statusData['absent']++;
                    }
                }
            }
            $currentDate->addDay();
        }

        // Calculate summary statistics
        $totalDays = count($attendanceData);
        $presentDays = $statusData['present'] + $statusData['late'];
        $attendanceRate = $totalDays > 0 ? round(($presentDays / $totalDays) * 100, 1) : 0;
        $avgHoursPerDay = count($hoursWorkedData) > 0 ? round(array_sum($hoursWorkedData) / count($hoursWorkedData), 2) : 0;
        $totalHours = array_sum($hoursWorkedData);

        return [
            'labels' => $labels,
            'datasets' => [
                'attendance' => $attendanceData,
                'hours_worked' => $hoursWorkedData,
            ],
            'summary' => [
                'attendance_rate' => $attendanceRate,
                'total_hours' => round($totalHours, 2),
                'avg_hours_per_day' => $avgHoursPerDay,
                'present_days' => $statusData['present'],
                'late_days' => $statusData['late'],
                'absent_days' => $statusData['absent'],
                'total_working_days' => $totalDays,
            ],
            'status_breakdown' => $statusData,
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d'),
                'type' => $type
            ]
        ];
    }

    private function getEmptyPerformanceData($type = 'week')
    {
        $labels = $type === 'week'
            ? ['Mon', 'Tue', 'Wed', 'Thu', 'Fri']
            : [];

        return [
            'labels' => $labels,
            'datasets' => [
                'attendance' => array_fill(0, count($labels), 0),
                'hours_worked' => array_fill(0, count($labels), 0),
            ],
            'summary' => [
                'attendance_rate' => 0,
                'total_hours' => 0,
                'avg_hours_per_day' => 0,
                'present_days' => 0,
                'late_days' => 0,
                'absent_days' => 0,
                'total_working_days' => 0,
            ],
            'status_breakdown' => ['present' => 0, 'late' => 0, 'absent' => 0],
            'period' => [
                'start' => null,
                'end' => null,
                'type' => $type
            ]
        ];
    }
}