<?php

namespace App\Services;

use App\Models\Department;
use Carbon\Carbon;

class ProfileTabService
{
    public function getNavigationTabs($user)
    {
        $tabsConfig = config('profile_tabs.tabs');
        $defaultActive = config('profile_tabs.default_active');
        $tabs = [];
        $isFirstTab = true;

        // Sort tabs by order
        uasort($tabsConfig, function($a, $b) {
            return ($a['order'] ?? 999) <=> ($b['order'] ?? 999);
        });

        foreach ($tabsConfig as $tabKey => $config) {
            if (!$config['enabled'] || !$this->hasPermissionForTab($user, $tabKey, $config['permissions'])) {
                continue;
            }

            $tabData = [
                'id' => $this->getTabId($tabKey),
                'label' => $config['label'],
                'value' => $this->getTabValue($user, $tabKey),
                'active' => ($tabKey === $defaultActive) || ($isFirstTab && $defaultActive === null),
                'icon' => $config['icon'] ?? 'fas fa-info',
                'order' => $config['order'] ?? 999,
                'css_class' => $config['css_class'] ?? '',
                'link_class' => $config['link_class'] ?? '',
                'value_class' => $config['value_class'] ?? '',
                'tooltip' => $config['tooltip'] ?? '',
            ];

            // Add URL routing
            if (isset($config['url_route']) && $config['url_route']) {
                $tabData['url'] = route($config['url_route']);
            }

            // Add AJAX loading
            if (isset($config['ajax_load']) && $config['ajax_load']) {
                $tabData['ajax_url'] = route('profile.tab.data', ['tab' => $tabKey]);
                $tabData['data_attributes'] = ['tab-key' => $tabKey];
            }

            // Add badge support
            if (isset($config['show_badge']) && $config['show_badge']) {
                $tabData['badge'] = $this->getTabBadge($user, $tabKey, $config['badge_type']);
            }

            // Add trend indicators
            if (isset($config['trend_enabled']) && $config['trend_enabled']) {
                $tabData['trend'] = $this->getTabTrend($user, $tabKey);
            }

            $tabs[] = $tabData;
            $isFirstTab = false;
        }

        return $tabs;
    }

    public function hasPermissionForTab($user, $tabType, $permissions = [])
    {
        if (empty($permissions) || in_array('*', $permissions)) {
            return true;
        }

        return in_array($user->role, $permissions);
    }

    private function getTabId($tabKey)
    {
        $idMap = [
            'attendance' => 'profile-attendance',
            'messages' => 'profile-mail',
            'team' => 'profile-departments',
            'announcements' => 'profile-posts',
            'performance' => 'profile-performance'
        ];

        return $idMap[$tabKey] ?? 'profile-' . $tabKey;
    }

    public function getTabValue($user, $tabKey)
    {
        switch ($tabKey) {
            case 'attendance':
                return $this->getAttendanceData($user);
            case 'messages':
                return $this->getMessagesCount($user);
            case 'team':
                return $this->getTeamCount($user);
            case 'announcements':
                return $this->getAnnouncementsCount($user);
            case 'performance':
                return $this->getPerformanceTabValue($user);
            default:
                return '0';
        }
    }

    private function getPerformanceTabValue($user)
    {
        if (!$user->employee) {
            return 'N/A';
        }

        $currentMonth = Carbon::now()->startOfMonth();
        $attendanceLogs = $user->employee->getAttendanceLogs()
            ->whereBetween('check_time', [$currentMonth, Carbon::now()])
            ->get();

        $dailyAttendance = $attendanceLogs->groupBy(function($log) {
            return $log->check_time->format('Y-m-d');
        });

        $presentDays = 0;
        foreach ($dailyAttendance as $logs) {
            $checkIns = $logs->where('type', 'check_in');
            if ($checkIns->isNotEmpty()) {
                $presentDays++;
            }
        }

        // Use diffInWeekdays correctly and ensure minimum 1 working day
        $workingDays = $currentMonth->diffInWeekdays(Carbon::now()->endOfDay()) + 1;
        $workingDays = max(1, $workingDays);

        $rate = $workingDays > 0 ? round(($presentDays / $workingDays) * 100) : 0;
        $rate = min(100, $rate); // Cap at 100%

        if ($rate >= 95) return 'A+';
        if ($rate >= 90) return 'A';
        if ($rate >= 85) return 'B+';
        if ($rate >= 80) return 'B';
        if ($rate >= 75) return 'C+';
        if ($rate >= 70) return 'C';
        return 'D';
    }

    private function getAttendanceData($user)
    {
        if (!$user->employee) {
            return '0%';
        }

        $employee = $user->employee;
        $currentMonth = Carbon::now()->startOfMonth();

        // Use AttendanceLog through employee relationship
        $attendanceLogs = $employee->getAttendanceLogs()
            ->whereBetween('check_time', [$currentMonth, Carbon::now()])
            ->get();

        // Group by date to count present days
        $dailyAttendance = $attendanceLogs->groupBy(function($log) {
            return $log->check_time->format('Y-m-d');
        });

        $presentDays = 0;
        foreach ($dailyAttendance as $date => $logs) {
            $checkIns = $logs->where('type', 'check_in');
            if ($checkIns->isNotEmpty()) {
                $presentDays++;
            }
        }

        // Calculate working days from start of month or hire date
        $hireDate = $employee->hire_date ? Carbon::parse($employee->hire_date) : $currentMonth;
        $startDate = $hireDate->greaterThan($currentMonth) ? $hireDate : $currentMonth;

        // Use diffInWeekdays correctly (it doesn't include the end date by default)
        $workingDays = $startDate->diffInWeekdays(Carbon::now()->endOfDay()) + 1;

        // Ensure we have at least 1 working day to avoid division issues
        $workingDays = max(1, $workingDays);

        // Calculate percentage and cap at 100%
        $percentage = $workingDays > 0 ? round(($presentDays / $workingDays) * 100) : 0;
        $percentage = min(100, $percentage); // Cap at 100%

        return $percentage . '%';
    }

    private function getMessagesCount($user)
    {
        // Implement based on your messaging system
        return 3;
    }

    private function getTeamCount($user)
    {
        if ($user->team_id) {
            return Department::find($user->team_id)?->teamMembers()->count() ?? 0;
        }
        return 0;
    }

    private function getAnnouncementsCount($user)
    {
        // Implement based on your announcements system
        return 4;
    }

    public function getTabBadge($user, $tabKey, $badgeType)
    {
        $badge = [
            'show' => false,
            'value' => '',
            'class' => 'bg-primary'
        ];

        switch ($tabKey) {
            case 'attendance':
                if ($badgeType === 'notification') {
                    $lateCount = $this->getLateAttendanceCount($user);
                    if ($lateCount > 0) {
                        $badge = [
                            'show' => true,
                            'value' => $lateCount,
                            'class' => 'bg-warning'
                        ];
                    }
                }
                break;

            case 'messages':
                if ($badgeType === 'count') {
                    $unreadCount = $this->getUnreadMessagesCount($user);
                    if ($unreadCount > 0) {
                        $badge = [
                            'show' => true,
                            'value' => $unreadCount > 99 ? '99+' : $unreadCount,
                            'class' => 'bg-danger'
                        ];
                    }
                }
                break;

            case 'announcements':
                if ($badgeType === 'notification') {
                    $newCount = $this->getNewAnnouncementsCount($user);
                    if ($newCount > 0) {
                        $badge = [
                            'show' => true,
                            'value' => 'New',
                            'class' => 'bg-info'
                        ];
                    }
                }
                break;
        }

        return $badge;
    }

    public function getTabTrend($user, $tabKey)
    {
        switch ($tabKey) {
            case 'attendance':
                return $this->getAttendanceTrend($user);
            case 'performance':
                return $this->getPerformanceTrend($user);
            default:
                return null;
        }
    }

    private function getAttendanceTrend($user)
    {
        $currentMonth = Carbon::now()->startOfMonth();
        $lastMonth = Carbon::now()->subMonth()->startOfMonth();

        $currentCount = \App\Models\Attendance::where('user_id', $user->id)
            ->where('check_in', '>=', $currentMonth)
            ->count();

        $lastMonthCount = \App\Models\Attendance::where('user_id', $user->id)
            ->whereBetween('check_in', [$lastMonth, $currentMonth])
            ->count();

        return $currentCount >= $lastMonthCount ? 'up' : 'down';
    }

    private function getPerformanceTrend($user)
    {
        // Implement based on your performance metrics
        return 'up'; // Default to positive trend
    }

    private function getLateAttendanceCount($user)
    {
        // Implement based on your attendance rules
        return 2; // Example: 2 late attendances
    }

    private function getUnreadMessagesCount($user)
    {
        // Implement based on your messaging system
        return 5; // Example: 5 unread messages
    }

    private function getNewAnnouncementsCount($user)
    {
        // Implement based on your announcements system
        return 1; // Example: 1 new announcement
    }

    public function getTabData($user, $tab)
    {
        $tabsConfig = config('profile_tabs.tabs');

        if (!isset($tabsConfig[$tab]) || !$tabsConfig[$tab]['enabled']) {
            return null;
        }

        $config = $tabsConfig[$tab];
        if (!$this->hasPermissionForTab($user, $tab, $config['permissions'])) {
            return null;
        }

        return [
            'value' => $this->getTabValue($user, $tab),
            'badge' => $config['show_badge'] ? $this->getTabBadge($user, $tab, $config['badge_type']) : null,
            'trend' => $config['trend_enabled'] ? $this->getTabTrend($user, $tab) : null,
        ];
    }
}