<?php

namespace App\Services;

use Illuminate\Http\Request;

/**
 * Class TrackingService
 *
 * Central point to merge common tracking info (IPs, UTM, referrer, page URL)
 * into validated payloads across BRNC controllers.
 */
class TrackingService
{
    /**
     * Merge tracking fields with validated data.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  array $validatedData
     * @return array
     */
    public static function collect(Request $request, array $validatedData): array
    {
        return array_merge($validatedData, [
            // Server observed IP
            'platform_ip_2' => $request->server('REMOTE_ADDR'),

            // UTM / Attribution
            'source'      => $request->input('source', 'direct'),
            'medium'      => $request->input('medium'),
            'campaign'    => $request->input('campaign'),
            'term'        => $request->input('term'),
            'content'     => $request->input('content'),
            'referrer'    => $request->input('referrer', $request->headers->get('referer')),
            'page_url'    => $request->input('page_url', url()->current()),
            'visitor_id'  => $request->input('visitor_id'),
        ]);
    }

    /**
     * Safely cast "account" for tables that expect integer account IDs.
     * Returns null if not numeric.
     */
    public static function castAccountId(?string $account): ?int
    {
        if (is_null($account)) return null;
        return ctype_digit($account) ? (int) $account : null;
    }
}
