<?php

namespace App\Services;

use App\Enums\VacationStatus;
use App\Models\User;
use App\Models\Vacation;
use App\Models\VacationType;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use App\Notifications\VacationStatusNotification;
use Illuminate\Validation\ValidationException;

class VacationService
{
    /**
     * احسب إجمالي الأيام، مع استثناء نهاية الأسبوع لو النوع يسمح بذلك.
     * weekend_days مكوّنة من ['sun','mon','tue','wed','thu','fri','sat']
     */
    public function calculateTotalDays(Carbon $start, Carbon $end, VacationType $type): int
    {
        $days = 0;

        // خريطة days-of-week من config إلى رقم PHP (0=الأحد .. 6=السبت)
        $map = [
            'sun' => 0,
            'mon' => 1,
            'tue' => 2,
            'wed' => 3,
            'thu' => 4,
            'fri' => 5,
            'sat' => 6,
        ];
        $weekendCfg = (array) config('attendance.weekend_days', ['fri', 'sat']);
        $weekendIdx = array_values(array_intersect_key($map, array_flip($weekendCfg)));

        $period = new \DatePeriod($start->copy()->startOfDay(), new \DateInterval('P1D'), $end->copy()->endOfDay()->addDay());
        foreach ($period as $date) {
            $dow = (int) $date->format('w');
            $isWeekend = in_array($dow, $weekendIdx, true);
            if ($type->weekend_excluded && $isWeekend) continue;
            $days++;
        }
        return max($days, 0);
    }

    /**
     * إنشاء طلب إجازة (يحدّد needs_hr تلقائيًا من config + نوع الإجازة).
     */
    public function create(array $data, int $employeeId, int $requestedByUserId): Vacation
    {
        return DB::transaction(function () use ($data, $employeeId, $requestedByUserId) {
            $type  = VacationType::findOrFail($data['vacation_type_id']);

            $start = Carbon::parse($data['start_date'])->startOfDay();
            $end   = Carbon::parse($data['end_date'])->endOfDay();

            $totalDays = $this->calculateTotalDays($start, $end, $type);
            if ($type->max_days_per_request && $totalDays > $type->max_days_per_request) {
                abort(422, 'عدد الأيام المطلوب يتجاوز الحد المسموح لهذا النوع.');
            }

            $needsHr = $this->computeNeedsHrOnCreate($type, $totalDays);

            $vac = Vacation::create([
                'employee_id'                 => $employeeId,
                'vacation_type_id'            => $type->id,
                'start_date'                  => $start->toDateString(),
                'end_date'                    => $end->toDateString(),
                'total_days'                  => $totalDays,
                'member_reason'               => $data['member_reason'] ?? null,
                'status'                      => VacationStatus::PENDING,
                'needs_hr'                    => $needsHr,
                'requested_by_user_id'        => $requestedByUserId,
                'leader_reason'               => null,
                'hr_reason'                   => null,
                'leader_approved_by_user_id'  => null,
                'hr_approved_by_user_id'      => null,
                'leader_decision_at'          => null,
                'hr_decision_at'              => null,
            ]);

            return $vac->refresh();
        });
    }

    /**
     * تعديل طلب (مسموح في Pending فقط حسب الـPolicy).
     * يُعاد احتساب needs_hr لو تغيّر النوع/المدّة.
     */
    public function update(Vacation $vacation, array $data): Vacation
    {
        return DB::transaction(function () use ($vacation, $data) {
            $type  = VacationType::findOrFail($data['vacation_type_id']);
            $start = Carbon::parse($data['start_date'])->startOfDay();
            $end   = Carbon::parse($data['end_date'])->endOfDay();

            $totalDays = $this->calculateTotalDays($start, $end, $type);
            if ($type->max_days_per_request && $totalDays > $type->max_days_per_request) {
                abort(422, 'عدد الأيام المطلوب يتجاوز الحد المسموح لهذا النوع.');
            }

            $needsHr = $this->computeNeedsHrOnCreate($type, $totalDays);

            $vacation->update([
                'vacation_type_id' => $type->id,
                'start_date'       => $start->toDateString(),
                'end_date'         => $end->toDateString(),
                'total_days'       => $totalDays,
                'member_reason'    => $data['member_reason'] ?? null,
                'needs_hr'         => $needsHr,
            ]);

            return $vacation->refresh();
        });
    }

    /**
     * اعتماد TL
     * - لازم الحالة Pending (مضمونة من Policy).
     * - يضبط الحالة LEADER_APPROVED (نهائية لو needs_hr=false).
     */
    public function leaderApprove(Vacation $vacation, ?string $reason = null, ?int $approverUserId = null): Vacation
    {
        if (!$approverUserId) abort(403, 'no approver provided.');

        return DB::transaction(function () use ($vacation, $reason, $approverUserId) {
            if ($vacation->status !== VacationStatus::PENDING) {
                abort(422, 'لا يمكن اعتماد طلب غير Pending بواسطة التيم ليدر.');
            }

            $vacation->update([
                'status'                     => VacationStatus::LEADER_APPROVED,
                'leader_reason'              => $reason,
                'leader_approved_by_user_id' => $approverUserId,
                'leader_decision_at'         => now(),
            ]);

            // 🔔 إشعار صاحب الطلب
            if ($user = optional($vacation->employee)->user) {
                $actor = User::find($approverUserId);
                $user->notify(new VacationStatusNotification($vacation->fresh(), 'leader_approved', $actor));
            }

            return $vacation->refresh();
        });
    }

    /**
     * اعتماد HR
     * - مسموح في PENDING أو LEADER_APPROVED.
     * - ممنوع بعد بداية يوم الإجازة (نافذة HR).
     */
    public function hrApprove(Vacation $vacation, ?string $reason = null, ?int $approverUserId = null): Vacation
    {
        if (!$approverUserId) abort(403, 'no approver provided.');

        return DB::transaction(function () use ($vacation, $reason, $approverUserId) {
            $allowedValues = [
                VacationStatus::PENDING->value,
                VacationStatus::LEADER_APPROVED->value,
            ];
            $current = $vacation->status->value ?? (string) $vacation->status;
            if (!in_array($current, $allowedValues, true)) {
                abort(422, 'لا يمكن اعتماد هذا الطلب بواسطة HR في حالته الحالية.');
            }

            $this->assertHrWindowOpen($vacation);

            $vacation->update([
                'status'                 => VacationStatus::HR_APPROVED,
                'hr_reason'              => $reason,
                'hr_approved_by_user_id' => $approverUserId,
                'hr_decision_at'         => now(),
                'needs_hr'               => false,
            ]);

            if ($user = optional($vacation->employee)->user) {
                $actor = User::find($approverUserId);
                $user->notify(new VacationStatusNotification($vacation->fresh(), 'hr_approved', $actor));
            }

            return $vacation->refresh();
        });
    }

    /**
     * رفض (TL أو HR)
     * - TL: مسموح في PENDING فقط.
     * - HR/Admin: مسموح في PENDING و LEADER_APPROVED، وبداخل نافذة HR.
     */
    public function reject(Vacation $vacation, ?string $reason = null, string $by = 'leader', ?int $approverUserId = null): Vacation
    {
        if (!$approverUserId) abort(403, 'no approver provided.');

        return DB::transaction(function () use ($vacation, $reason, $by, $approverUserId) {
            $by = strtolower($by);

            if ($by === 'hr') {
                $allowedValues = [
                    VacationStatus::PENDING->value,
                    VacationStatus::LEADER_APPROVED->value,
                ];
                $current = $vacation->status->value ?? (string) $vacation->status;
                if (!in_array($current, $allowedValues, true)) {
                    abort(422, 'لا يمكن رفض هذا الطلب بواسطة HR في حالته الحالية.');
                }

                $this->assertHrWindowOpen($vacation);

                $vacation->update([
                    'status'                 => VacationStatus::HR_REJECTED,
                    'hr_reason'              => $reason,
                    'hr_approved_by_user_id' => $approverUserId,
                    'hr_decision_at'         => now(),
                    'needs_hr'               => false,
                ]);

                if ($user = optional($vacation->employee)->user) {
                    $actor = User::find($approverUserId);
                    $user->notify(new VacationStatusNotification($vacation->fresh(), 'hr_rejected', $actor));
                }
            } else { // leader
                if ($vacation->status !== VacationStatus::PENDING) {
                    throw ValidationException::withMessages(['status' => 'فقط الطلبات Pending يمكن رفضها بواسطة TL.']);
                }

                $vacation->update([
                    'status'                     => VacationStatus::LEADER_REJECTED,
                    'leader_reason'              => $reason,
                    'leader_approved_by_user_id' => $approverUserId,
                    'leader_decision_at'         => now(),
                    'needs_hr'                   => false,
                ]);

                if ($user = optional($vacation->employee)->user) {
                    $actor = User::find($approverUserId);
                    $user->notify(new VacationStatusNotification($vacation->fresh(), 'leader_rejected', $actor));
                }
            }

            return $vacation->refresh();
        });
    }

    /**
     * إلغاء بواسطة صاحب الطلب (بديل عن delete الفعلي).
     * Policy تضمن Owner + Pending.
     */
    public function cancelByMember(Vacation $vacation): Vacation
    {
        $vacation->update([
            'status' => VacationStatus::CANCELLED,
        ]);
        return $vacation->refresh();
    }

    /**
     * (اختياري) إنشاء لعضو بواسطة الإدارة
     */
    public function createForEmployee(array $data, int $employeeId): Vacation
    {
        return DB::transaction(function () use ($data, $employeeId) {
            $type  = VacationType::findOrFail($data['vacation_type_id']);
            $start = Carbon::parse($data['start_date'])->startOfDay();
            $end   = Carbon::parse($data['end_date'])->endOfDay();

            $totalDays = $this->calculateTotalDays($start, $end, $type);
            if ($type->max_days_per_request && $totalDays > $type->max_days_per_request) {
                abort(422, 'عدد الأيام المطلوب يتجاوز الحد المسموح لهذا النوع.');
            }

            $needsHr = $this->computeNeedsHrOnCreate($type, $totalDays);

            $vac = Vacation::create([
                'employee_id'      => $employeeId,
                'vacation_type_id' => $type->id,
                'start_date'       => $start->toDateString(),
                'end_date'         => $end->toDateString(),
                'total_days'       => $totalDays,
                'member_reason'    => $data['member_reason'] ?? null,
                'status'           => VacationStatus::PENDING,
                'needs_hr'         => $needsHr,
            ]);

            return $vac->refresh();
        });
    }

    /* ==================== Helpers ==================== */

    /**
     * احتساب الاحتياج لـ HR بناءً على:
     * - config: attendance.vacations.needs_hr_if_days_over
     * - config: attendance.vacations.types_need_hr (slug/name)
     * - نوع الإجازة: required_approvals === 2
     */
    protected function computeNeedsHrOnCreate(VacationType $type, int $days): bool
    {
        $limitDays   = (int) config('attendance.vacations.needs_hr_if_days_over', 5);
        $typesNeedHr = (array) config('attendance.vacations.types_need_hr', []);

        if ($days > $limitDays) return true;

        // لو النوع له slug/name مطابق لقائمة types_need_hr
        $slug = $type->slug ?? null;
        $name = $type->name ?? null;
        if ($slug && in_array($slug, $typesNeedHr, true)) return true;
        if ($name && in_array($name, $typesNeedHr, true)) return true;

        // fallback: حسب تعريف النوع
        if ((int) ($type->required_approvals ?? 1) === 2) return true;

        return false;
    }

    /**
     * HR نافذته مفتوحة حتى بداية يوم الإجازة.
     */
    protected function assertHrWindowOpen(Vacation $v): void
    {
        $now = now();
        $deadline = $v->start_date instanceof Carbon
            ? $v->start_date->copy()->startOfDay()
            : Carbon::parse($v->start_date)->startOfDay();

        // HR لا يقرّر بعد بداية يوم الإجازة
        if ($now->greaterThanOrEqualTo($deadline)) {
            abort(422, 'انتهت نافذة قرارات HR بعد بداية الإجازة.');
        }
    }
}
