<?php
declare(strict_types=1);

namespace App\Support;

use Carbon\Carbon;

/**
 * MailFmt
 * Utilities to normalize Webklex PHPIMAP attributes across versions.
 * - Safe array casting for Attribute/Collection/Iterator
 * - Address list to string
 * - Safe string/date extraction
 * - Flags & attachments compatibility
 */
class MailFmt
{
    /** Normalize Attribute/Collection/Iterator/array to array */
    public static function toArray(mixed $attr): array {
        if ($attr === null) return [];
        if (is_array($attr)) return $attr;
        if ($attr instanceof \Traversable) return iterator_to_array($attr);

        if (is_object($attr)) {
            if (method_exists($attr, 'toArray')) {
                $arr = $attr->toArray();
                return is_array($arr) ? $arr : ( $arr instanceof \Traversable ? iterator_to_array($arr) : [] );
            }
            if (method_exists($attr, 'all')) {
                $arr = $attr->all();
                return is_array($arr) ? $arr : [];
            }
            if (method_exists($attr, 'get')) {
                $v = $attr->get();
                if (is_array($v)) return $v;
                if ($v instanceof \Traversable) return iterator_to_array($v);
                if ($v !== null) return [$v];
            }
        }
        return [];
    }

    /** Convert an addresses list (any shape) to "a@b.com, c@d.com" */
    public static function addressesToString(mixed $attr): string {
        $list = self::toArray($attr);
        $out = [];

        foreach ($list as $a) {
            $email = null;

            if (is_object($a)) {
                if (property_exists($a, 'mail') && $a->mail) {
                    $email = $a->mail;
                } elseif (property_exists($a, 'mailbox') || property_exists($a, 'host')) {
                    $local = $a->mailbox ?? null;
                    $host  = $a->host ?? null;
                    if ($local && $host) $email = $local.'@'.$host;
                } elseif (method_exists($a, 'get')) {
                    $v = $a->get();
                    if (is_string($v)) $email = $v;
                } elseif ($a instanceof \Stringable) {
                    $email = (string) $a;
                }
            } elseif (is_string($a)) {
                $email = $a;
            } elseif (is_array($a)) {
                $email = $a['mail']
                    ?? ($a['address'] ?? (isset($a['mailbox'],$a['host']) ? $a['mailbox'].'@'.$a['host'] : null));
            }

            if ($email) $out[] = $email;
        }

        return implode(', ', $out);
    }

    /** Safely convert Attribute/Stringable to string */
    public static function string(mixed $attr): ?string {
        if ($attr === null) return null;
        if (is_string($attr)) return $attr;
        if ($attr instanceof \Stringable) return (string) $attr;
        if (is_object($attr) && method_exists($attr, 'get')) {
            $v = $attr->get();
            if (is_string($v)) return $v;
            if ($v instanceof \Stringable) return (string) $v;
        }
        return null;
    }

    /** Safely format an Attribute/Date to Y-m-d H:i (or custom) */
    public static function date(mixed $attr, string $fmt='Y-m-d H:i'): ?string {
        if ($attr === null) return null;
        if ($attr instanceof \DateTimeInterface) return $attr->format($fmt);

        if (is_object($attr) && method_exists($attr, 'toDate')) {
            $dt = $attr->toDate();
            if ($dt instanceof \DateTimeInterface) return $dt->format($fmt);
        }
        if (is_object($attr) && method_exists($attr, 'get')) {
            $raw = $attr->get();
            if ($raw instanceof \DateTimeInterface) return $raw->format($fmt);
            if (is_string($raw) && $raw !== '') {
                try { return Carbon::parse($raw)->format($fmt); } catch (\Throwable) { return $raw; }
            }
        }
        if (is_string($attr) && $attr !== '') {
            try { return Carbon::parse($attr)->format($fmt); } catch (\Throwable) { return $attr; }
        }
        return null;
    }

    /** Seen flag compatibility across versions */
    public static function isSeen(object $msg): bool {
        try { if (method_exists($msg, 'isSeen')) return (bool) $msg->isSeen(); } catch (\Throwable) {}
        try {
            if (method_exists($msg, 'hasFlag')) {
                return $msg->hasFlag('\Seen') || $msg->hasFlag('Seen') || $msg->hasFlag('SEEN');
            }
            if (method_exists($msg, 'getFlags')) {
                $flags = $msg->getFlags();
                // collection-like
                if (is_object($flags) && method_exists($flags, 'contains')) {
                    return $flags->contains('\Seen') || $flags->contains('Seen') || $flags->contains('SEEN');
                }
                if (is_object($flags) && method_exists($flags, 'has')) {
                    return $flags->has('\Seen') || $flags->has('Seen') || $flags->has('SEEN');
                }
                if (is_object($flags) && method_exists($flags, 'get')) {
                    $raw = $flags->get();
                    if (is_array($raw)) {
                        return in_array('\Seen', $raw, true) || in_array('Seen', $raw, true) || in_array('SEEN', $raw, true);
                    }
                }
                if (is_array($flags)) {
                    return in_array('\Seen', $flags, true) || in_array('Seen', $flags, true) || in_array('SEEN', $flags, true);
                }
            }
        } catch (\Throwable) {}
        return false;
    }

    /** Attachments compat (true if any) */
    public static function hasAttachments(object $msg): bool {
        try { if (method_exists($msg, 'hasAttachments')) return (bool) $msg->hasAttachments(); } catch (\Throwable) {}
        try {
            if (method_exists($msg, 'getAttachments')) {
                $atts = $msg->getAttachments();
                if (is_iterable($atts)) {
                    foreach ($atts as $_) { return true; }
                }
            }
        } catch (\Throwable) {}
        return false;
    }
}
