<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {

    public function up(): void
    {
        // 0) BRANCHES
        if (!Schema::hasTable('branches')) {
            Schema::create('branches', function (Blueprint $table) {
                $table->id(); // PK
                $table->integer('branch_id')->unique(); // الكود القادم من الأجهزة/النُظم
                $table->string('name');
                $table->string('location')->nullable();
                $table->timestamps();
                $table->index('name', 'idx_branches_name');
            });
        }

        // 1) TEAM MEMBERS
        Schema::create('team_members', function (Blueprint $t) {
            $t->id();

            $t->string('employee_code'); // من غير unique لوحده
            $t->string('first_name')->nullable();
            $t->string('last_name')->nullable();
            $t->string('name')->nullable();
            $t->enum('role', ['admin','manager','member'])->default('member');

            $t->unsignedBigInteger('department_id')->nullable();
            $t->unsignedBigInteger('manager_id')->nullable();

            // FK إلى branches.id (مش branches.branch_id)
            $t->unsignedBigInteger('branch_id')->nullable();

            $t->string('email')->nullable();
            $t->string('phone')->nullable();
            $t->date('hire_date')->nullable();
            $t->enum('status', ['active','inactive'])->default('active');

            $t->timestamps();
            $t->softDeletes();

            $t->index('department_id', 'idx_tm_department_id');
            $t->index('manager_id',   'idx_tm_manager_id');
            $t->index('status',       'idx_tm_status');
            $t->index('branch_id',    'idx_tm_branch_id');

            // مفتاح مركّب (employee_code + branch_id[id])
            $t->unique(['employee_code', 'branch_id'], 'uq_tm_code_branch');
        });

        Schema::table('team_members', function (Blueprint $t) {
            if (Schema::hasTable('departments')) {
                $t->foreign('department_id', 'fk_tm_department')
                  ->references('id')->on('departments')
                  ->nullOnDelete();
            }

            $t->foreign('manager_id', 'fk_tm_manager_self')
              ->references('id')->on('team_members')
              ->nullOnDelete();

            // ربط بــ branches.id (مش branch_id)
            if (Schema::hasTable('branches')) {
                $t->foreign('branch_id', 'fk_tm_branch')
                  ->references('id')->on('branches')
                  ->nullOnDelete();
            }
        });

        // 2) ALTER DEPARTMENTS
        if (Schema::hasTable('departments')) {
            Schema::table('departments', function (Blueprint $t) {
                if (!Schema::hasColumn('departments', 'manager_id')) {
                    $t->unsignedBigInteger('manager_id')->nullable()->after('parent_id');
                    $t->index('manager_id', 'idx_dept_manager_id');
                }
            });

            Schema::table('departments', function (Blueprint $t) {
                $t->foreign('manager_id', 'fk_departments_manager_tm_v1')
                  ->references('id')->on('team_members')
                  ->nullOnDelete();
            });
        }

        // 3) TEAMS
        Schema::create('teams', function (Blueprint $t) {
            $t->id();
            $t->string('name')->unique();
            $t->enum('type', ['squad','project','departmental'])->default('squad');
            $t->unsignedBigInteger('account_manager_id')->nullable();
            $t->date('start_date')->nullable();
            $t->date('end_date')->nullable();
            $t->boolean('is_active')->default(true);
            $t->timestamps();
            $t->softDeletes();

            $t->index('type', 'idx_teams_type');
            $t->index('is_active', 'idx_teams_active');
            $t->index('account_manager_id', 'idx_teams_am_id');
        });

        Schema::table('teams', function (Blueprint $t) {
            $t->foreign('account_manager_id', 'fk_teams_account_manager_tm_v1')
              ->references('id')->on('team_members')
              ->nullOnDelete();
        });

        // 4) TEAM MEMBERSHIPS
        Schema::create('team_memberships', function (Blueprint $t) {
            $t->id();
            $t->unsignedBigInteger('team_id');
            $t->unsignedBigInteger('team_member_id');
            $t->enum('role_in_team', [
                'member','lead','account_manager','media_buyer','designer','editor','content','developer'
            ])->default('member');
            $t->date('joined_at')->nullable();
            $t->date('left_at')->nullable();
            $t->boolean('is_active')->default(true);
            $t->timestamps();

            $t->index(['team_id', 'team_member_id'], 'idx_tms_team_member');
            $t->index('is_active', 'idx_tms_active');
            $t->unique(['team_id','team_member_id','joined_at'], 'uq_tms_team_member_joined');
        });

        Schema::table('team_memberships', function (Blueprint $t) {
            $t->foreign('team_id', 'fk_tms_team_v1')
              ->references('id')->on('teams')
              ->cascadeOnDelete();

            $t->foreign('team_member_id', 'fk_tms_member_v1')
              ->references('id')->on('team_members')
              ->cascadeOnDelete();
        });
    }

    public function down(): void
    {
        if (Schema::hasTable('team_memberships')) {
            Schema::table('team_memberships', function (Blueprint $t) {
                try { $t->dropForeign('fk_tms_team_v1'); } catch (\Throwable $e) {}
                try { $t->dropForeign('fk_tms_member_v1'); } catch (\Throwable $e) {}
            });
            Schema::dropIfExists('team_memberships');
        }

        if (Schema::hasTable('teams')) {
            Schema::table('teams', function (Blueprint $t) {
                try { $t->dropForeign('fk_teams_account_manager_tm_v1'); } catch (\Throwable $e) {}
            });
            Schema::dropIfExists('teams');
        }

        if (Schema::hasTable('departments')) {
            Schema::table('departments', function (Blueprint $t) {
                try { $t->dropForeign('fk_departments_manager_tm_v1'); } catch (\Throwable $e) {}
                if (Schema::hasColumn('departments','manager_id')) {
                    try { $t->dropIndex('idx_dept_manager_id'); } catch (\Throwable $e) {}
                    $t->dropColumn('manager_id');
                }
            });
        }

        if (Schema::hasTable('team_members')) {
            Schema::table('team_members', function (Blueprint $t) {
                try { $t->dropForeign('fk_tm_department'); } catch (\Throwable $e) {}
                try { $t->dropForeign('fk_tm_manager_self'); } catch (\Throwable $e) {}
                try { $t->dropForeign('fk_tm_branch'); } catch (\Throwable $e) {}
                try { $t->dropIndex('idx_tm_department_id'); } catch (\Throwable $e) {}
                try { $t->dropIndex('idx_tm_manager_id'); } catch (\Throwable $e) {}
                try { $t->dropIndex('idx_tm_status'); } catch (\Throwable $e) {}
                try { $t->dropIndex('idx_tm_branch_id'); } catch (\Throwable $e) {}
                try { $t->dropUnique('uq_tm_code_branch'); } catch (\Throwable $e) {}
            });
            Schema::dropIfExists('team_members');
        }
        // branches: اتركه كما هو (أو احذفه لو أنت أنشأته هنا فقط)
        // Schema::dropIfExists('branches');
    }
};
