<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Arr;
use Faker\Factory as Faker;

use App\Models\AttendancePermit;
use App\Models\AttendancePermitType;
use App\Models\Employee;
use App\Models\User;

// لو بتستخدم Spatie Roles (اختياري للفول باك)
use Spatie\Permission\Models\Role;

class AttendancePermitsDemoSeeder extends Seeder
{
    /**
     * إعدادات مرنة
     */
    const PER_EMPLOYEE_MIN = 3;      // أقل عدد أذونات لكل موظف
    const PER_EMPLOYEE_MAX = 6;      // أقصى عدد أذونات لكل موظف
    const DATE_RANGE_DAYS  = 45;     // يتم توليد الأذونات خلال +/- عدد الأيام من اليوم
    const FORCE_SEED       = false;  // set true لو عايز تseed حتى لو الجدول فيه بيانات

    public function run(): void
    {
        if (!self::FORCE_SEED && AttendancePermit::query()->exists()) {
            $this->command->warn('attendance_permits already has data. Skipping (set FORCE_SEED=true to override).');
            return;
        }

        $faker = Faker::create('ar_EG');

        // ====== 1) تجهيز أنواع الأذونات ======
        $typesByName = AttendancePermitType::query()
            ->get()
            ->keyBy(fn($t) => strtolower($t->name));

        // fallback لو ناقص أنواع
        if ($typesByName->isEmpty()) {
            $this->command->error('No attendance_permit_types found. Seed AttendancePermitTypesSeeder first.');
            return;
        }

        // أسماء مرجّحة للاستخدام
        $preferredTypeNames = ['short_leave','early_exit','late_entry','medical','personal'];
        $availableTypeIds = [];
        foreach ($preferredTypeNames as $n) {
            $tt = $typesByName->get(strtolower($n));
            if ($tt) $availableTypeIds[] = $tt->id;
        }
        if (empty($availableTypeIds)) {
            // لو مفيش ولا واحد من الأسماء المفضلة، استخدم أي نوع موجود
            $availableTypeIds = $typesByName->pluck('id')->all();
        }

        // ====== 2) جهّز Users للأدوار (اختياري) ======
        $leaders = $this->usersForRole('TeamLeader');
        $hrs     = $this->usersForRole('HR');

        // fallback users (لو مفيش TeamLeader/HR)
        $anyUsers = User::query()->limit(10)->get();

        // ====== 3) لف على مجموعة موظفين معقولة ======
        $employees = Employee::query()->limit(25)->get(); // عدّل العدد كما تريد
        if ($employees->isEmpty()) {
            $this->command->error('No employees found. Add some employees first.');
            return;
        }

        // امسح الجدول أو لا حسب FORCE_SEED
        if (self::FORCE_SEED) {
            DB::table('attendance_permits')->truncate();
        }

        $rows = [];
        $now = now();

        foreach ($employees as $employee) {
            $count = random_int(self::PER_EMPLOYEE_MIN, self::PER_EMPLOYEE_MAX);

            for ($i = 0; $i < $count; $i++) {
                // 3.1 اختار نوع الإذن
                $permitTypeId = Arr::random($availableTypeIds);
                $typeObj = AttendancePermitType::find($permitTypeId);
                $typeName = $typeObj?->name ?? null;

                // 3.2 اختر تاريخ داخل نافذة زمنية
                // مزيج من الماضي والقريب/المستقبل
                $baseDay = Carbon::now()->addDays(random_int(-self::DATE_RANGE_DAYS, self::DATE_RANGE_DAYS));

                // احتمال قليل All-day لبعض الأنواع (مثلاً personal/medical)
                $isAllDay = false;
                if (in_array($typeName, ['personal','medical'], true)) {
                    $isAllDay = (bool) random_int(0, 5) === 0; // ~16% احتمالية
                }

                // 3.3 توليد فترة
                if ($isAllDay) {
                    $start = $baseDay->copy()->startOfDay()->addHours(9); // يبدأ 9 ص (شكلًا)
                    $end   = $baseDay->copy()->endOfDay()->subHours(1);  // ينتهي 11 م
                } else {
                    // إذن داخل اليوم: من 30 إلى 180 دقيقة، بين 8 ص و6 م
                    $startHour = random_int(8, 16); // بدء حتى 4 مساءً
                    $startMin  = [0, 15, 30, 45][array_rand([0,1,2,3])];
                    $duration  = [30, 45, 60, 90, 120, 150, 180][array_rand([0,1,2,3,4,5,6])];

                    $start = $baseDay->copy()->setTime($startHour, $startMin);
                    $end   = $start->copy()->addMinutes($duration);
                }

                $totalMinutes = $start->diffInMinutes($end);

                // 3.4 needs_hr: أغلبها true
                $needsHr = (bool) (random_int(0, 10) <= 7);

                // 3.5 الحالة (مع منطق تعبئة الحقول)
                $status = Arr::random([
                    'pending',
                    'leader_approved',
                    'hr_approved',
                    'rejected',
                    'cancelled',
                ]);

                // 3.6 requested_by_user_id من employee->user_id (لو موجود) وإلا أي حد
                $requestedById = null;
                if (isset($employee->user_id) && !empty($employee->user_id)) {
                    $requestedById = (int) $employee->user_id;
                } elseif ($anyUsers->isNotEmpty()) {
                    $requestedById = (int) Arr::random($anyUsers->pluck('id')->all());
                }

                // 3.7 تحديد الـ actors حسب الحالة
                $leaderId = $leaders->isNotEmpty()
                    ? (int) Arr::random($leaders->pluck('id')->all())
                    : ($anyUsers->isNotEmpty() ? (int) Arr::random($anyUsers->pluck('id')->all()) : null);

                $hrId = $hrs->isNotEmpty()
                    ? (int) Arr::random($hrs->pluck('id')->all())
                    : ($anyUsers->isNotEmpty() ? (int) Arr::random($anyUsers->pluck('id')->all()) : null);

                $leaderDecisionAt = null;
                $hrDecisionAt     = null;
                $leaderReason     = null;
                $hrReason         = null;
                $cancelledById    = null;
                $cancelledReason  = null;

                switch ($status) {
                    case 'pending':
                        // لا قرارات بعد
                        break;

                    case 'leader_approved':
                        $leaderDecisionAt = $start->copy()->subDays(random_int(1, 5));
                        $leaderReason     = $faker->realText(25);
                        // ممكن يبقى نهائي فعليًا لو needs_hr=false، لكن الحالة تبقى leader_approved
                        break;

                    case 'hr_approved':
                        // لازم نعدي على القائد منطقيًا
                        $leaderDecisionAt = $start->copy()->subDays(random_int(3, 7));
                        $leaderReason     = $faker->realText(20);
                        $hrDecisionAt     = $start->copy()->subDays(random_int(1, 2));
                        $hrReason         = $faker->realText(20);
                        break;

                    case 'rejected':
                        // الرفض ممكن من القائد أو HR
                        if (random_int(0,1) === 0) {
                            // رفض قائد
                            $leaderDecisionAt = $start->copy()->subDays(random_int(1, 5));
                            $leaderReason     = $faker->realText(20);
                            $hrId             = null; // لم يصل HR
                        } else {
                            // رفض HR (مرّ على القائد)
                            $leaderDecisionAt = $start->copy()->subDays(random_int(3, 7));
                            $hrDecisionAt     = $start->copy()->subDays(random_int(1, 2));
                            $leaderReason     = $faker->realText(15);
                            $hrReason         = $faker->realText(15);
                        }
                        break;

                    case 'cancelled':
                        // طلب اتلغى قبل الحسم النهائي
                        $cancelledById   = $requestedById ?: ($anyUsers->isNotEmpty() ? (int) Arr::random($anyUsers->pluck('id')->all()) : null);
                        $cancelledReason = $faker->sentence(6);
                        break;
                }

                $rows[] = [
                    'employee_id'                 => $employee->id,
                    'permit_type_id'              => $permitTypeId,

                    'requested_by_user_id'        => $requestedById,
                    'leader_approved_by_user_id'  => $leaderId,
                    'hr_approved_by_user_id'      => $hrId,

                    'start_datetime'              => $start->toDateTimeString(),
                    'end_datetime'                => $end->toDateTimeString(),
                    'all_day'                     => $isAllDay,
                    'total_minutes'               => $totalMinutes,

                    'member_reason'               => $faker->optional(0.7)->realText(40),
                    'leader_reason'               => $leaderReason,
                    'hr_reason'                   => $hrReason,

                    'status'                      => $status,
                    'needs_hr'                    => $needsHr,

                    'leader_decision_at'          => $leaderDecisionAt?->toDateTimeString(),
                    'hr_decision_at'              => $hrDecisionAt?->toDateTimeString(),

                    'cancelled_by_user_id'        => $cancelledById,
                    'cancelled_reason'            => $cancelledReason,

                    'created_at'                  => $now->copy()->subDays(random_int(0, 30)),
                    'updated_at'                  => $now,
                ];
            }
        }

        // ====== 4) إدخال البيانات ======
        // chunk insert for performance
        foreach (array_chunk($rows, 500) as $chunk) {
            AttendancePermit::insert($chunk);
        }

        $this->command->info('AttendancePermitsDemoSeeder: inserted '.count($rows).' permits for '.count($employees).' employees.');
    }

    /**
     * حاول تجيب مستخدمين لهم دور معيّن (Spatie).
     * لو الدور غير موجود أو فاضي، رجّع Collection فاضية.
     */
    protected function usersForRole(string $roleName)
    {
        try {
            $role = Role::where('name', $roleName)->first();
            if (!$role) return collect();
            // عبر relation users() لو متاحة
            if (method_exists($role, 'users')) {
                return $role->users()->limit(50)->get();
            }
        } catch (\Throwable $e) {
            // Spatie مش متاحة؟ عادي
        }
        return collect();
    }
}
