<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Employee;
use App\Models\Vacation;
use App\Models\VacationType;
use App\Enums\VacationStatus;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class VacationSeeder extends Seeder
{
    public function run(): void
    {
        // === 1. Vacation Types ===
        $types = [
            [
                'code' => 'annual',
                'name' => 'إجازة سنوية',
                'weekend_excluded' => true,
                'max_days_per_request' => 14,
                'required_approvals' => 2
            ],
            [
                'code' => 'sick',
                'name' => 'إجازة مرضية',
                'weekend_excluded' => false,
                'max_days_per_request' => 7,
                'required_approvals' => 1
            ],
            [
                'code' => 'emergency',
                'name' => 'إجازة طارئة',
                'weekend_excluded' => false,
                'max_days_per_request' => 3,
                'required_approvals' => 2
            ],
        ];

        foreach ($types as $t) {
            VacationType::firstOrCreate(['code' => $t['code']], $t);
        }

        // === 2. Users + Employees ===
        $users = [
            ['name' => 'موظف', 'email' => 'employee@test.com', 'role' => 'Employee'],
            ['name' => 'تيم ليدر', 'email' => 'leader@test.com', 'role' => 'TeamLeader'],
            ['name' => 'الموارد البشرية', 'email' => 'hr@test.com', 'role' => 'HR'],
            ['name' => 'أدمن', 'email' => 'admin@test.com', 'role' => 'Admin'],
        ];

        $userModels = [];
        foreach ($users as $u) {
            $user = User::firstOrCreate(
                ['email' => $u['email']],
                [
                    'name' => $u['name'],
                    'password' => Hash::make('password'),
                ]
            );

            if (method_exists($user, 'syncRoles')) {
                $user->syncRoles([$u['role']]);
            }

            $employee = Employee::firstOrCreate(
                ['user_id' => $user->id],
                [
                    'first_name' => $u['name'],
                    'last_name' => 'تجريبي',
                    'employee_code' => 'EMP-' . Str::upper(Str::random(6)),
                    'branch_id'     => 101,

                    ]
            );

            $userModels[$u['role']] = $user;
        }

        $employee = $userModels['Employee']->employee;
        $leader   = $userModels['TeamLeader'];
        $hr       = $userModels['HR'];

        $vacTypeAnnual = VacationType::where('code', 'annual')->first();

        // === 3. Vacations (تجريبية) ===
        // Pending
        Vacation::create([
            'employee_id' => $employee->id,
            'vacation_type_id' => $vacTypeAnnual->id,
            'start_date' => Carbon::today()->toDateString(),
            'end_date'   => Carbon::today()->addDays(5)->toDateString(),
            'total_days' => 5,
            'member_reason' => 'إجازة عائلية',
            'status' => VacationStatus::Pending,
            'needs_hr' => true,
            'requested_by_user_id' => $employee->user_id,
        ]);

        // Leader Approved
        Vacation::create([
            'employee_id' => $employee->id,
            'vacation_type_id' => $vacTypeAnnual->id,
            'start_date' => Carbon::today()->subDays(10)->toDateString(),
            'end_date'   => Carbon::today()->subDays(5)->toDateString(),
            'total_days' => 5,
            'member_reason' => 'سفر',
            'status' => VacationStatus::LeaderApproved,
            'needs_hr' => true,
            'requested_by_user_id' => $employee->user_id,
            'leader_reason' => 'موافقة لعدم وجود تضارب',
            'leader_approved_by_user_id' => $leader->id,
            'leader_decision_at' => now()->subDays(9),
        ]);

        // HR Approved
        Vacation::create([
            'employee_id' => $employee->id,
            'vacation_type_id' => $vacTypeAnnual->id,
            'start_date' => Carbon::today()->subDays(20)->toDateString(),
            'end_date'   => Carbon::today()->subDays(15)->toDateString(),
            'total_days' => 5,
            'member_reason' => 'ظروف عائلية',
            'status' => VacationStatus::HrApproved,
            'needs_hr' => true,
            'requested_by_user_id' => $employee->user_id,
            'leader_reason' => 'تمت الموافقة',
            'leader_approved_by_user_id' => $leader->id,
            'leader_decision_at' => now()->subDays(19),
            'hr_reason' => 'لا مانع',
            'hr_approved_by_user_id' => $hr->id,
            'hr_decision_at' => now()->subDays(18),
        ]);

        // Rejected
        Vacation::create([
            'employee_id' => $employee->id,
            'vacation_type_id' => $vacTypeAnnual->id,
            'start_date' => Carbon::today()->subDays(30)->toDateString(),
            'end_date'   => Carbon::today()->subDays(25)->toDateString(),
            'total_days' => 5,
            'member_reason' => 'خارج الجدول',
            'status' => VacationStatus::Rejected,
            'needs_hr' => true,
            'requested_by_user_id' => $employee->user_id,
            'leader_reason' => 'رفض بسبب تضارب',
            'leader_approved_by_user_id' => $leader->id,
            'leader_decision_at' => now()->subDays(29),
        ]);
    }
}
