<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use App\Models\User;

class RolePermissionSeeder extends Seeder
{
    public function run(): void
    {
        // مهم جداً: نفس الجارد المستخدم في المشروع (web)
        $guard = config('auth.defaults.guard', 'web');

        // امسح كاش الصلاحيات (لو متخزن)
        app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();

        // ===== تعريف الصلاحيات (من السايدبار والمنطق الحالي) =====
        $permissions = [
            // System / Roles
            'roles.manage',

            // Organization / Teams
            'teams.view',

            // Attendance (لوحة HR أو مشاهدة الجميع)
            'attendance.hr',
            'attendance.view.all',

            // Vacations (مثال صلاحيات CRUD/Workflow)
            'vacations.view',
            'vacations.create',
            'vacations.update',
            'vacations.delete',
            'vacations.approve.leader',
            'vacations.approve.hr',
            'vacations.reject',

            // Permits (الأذون)
            'permits.view',
            'permits.create',
            'permits.update',
            'permits.delete',
            'permits.approve.leader',
            'permits.approve.hr',
            'permits.cancel',

            // Email
            'email.inbox.view',
            'email.compose.send',
            'email.tools.health',
            'email.tools.dns',

            // Telescope (لو عايز تقفله على أدمن فقط)
            'telescope.view',
        ];

        foreach ($permissions as $p) {
            Permission::firstOrCreate(['name' => $p, 'guard_name' => $guard]);
        }

        // ===== تعريف الأدوار =====
        $roles = [
            'Admin' => [
                // كل الصلاحيات
                '*',
            ],

            'Team Lead' => [
                'teams.view',
                // Vacations
                'vacations.view',
                'vacations.create',
                'vacations.update',
                'vacations.approve.leader',
                'vacations.reject',
                // Permits
                'permits.view',
                'permits.create',
                'permits.update',
                'permits.approve.leader',
                // Email (لو حابب يطلع على Inbox فقط)
                'email.inbox.view',
            ],

            'HR Manager' => [
                'attendance.hr',
                'attendance.view.all',
                // Vacations
                'vacations.view',
                'vacations.approve.hr',
                'vacations.reject',
                // Permits
                'permits.view',
                'permits.approve.hr',
                // Email tools
                'email.tools.health',
                'email.tools.dns',
            ],

            'Employee' => [
                // الأقل صلاحيات
                'vacations.view',
                'vacations.create',
                'vacations.update',
                'permits.view',
                'permits.create',
                'permits.update',
                'permits.cancel',
                'email.inbox.view',
            ],
        ];

        foreach ($roles as $roleName => $rolePerms) {
            $role = Role::firstOrCreate(['name' => $roleName, 'guard_name' => $guard]);

            if ($rolePerms === ['*']) {
                // أدمن = كل الصلاحيات
                $role->syncPermissions(Permission::pluck('name'));
            } else {
                $role->syncPermissions($rolePerms);
            }
        }

        // ===== إنشاء يوزر أدمن للتجربة (غيّر الإيميل والباسورد بعد كده) =====
        $adminEmail = env('SEED_ADMIN_EMAIL', 'admin@example.com');

        $admin = User::firstOrCreate(
            ['email' => $adminEmail],
            [
                'name'     => 'System Admin',
                'password' => Hash::make(env('SEED_ADMIN_PASSWORD', 'password')),
            ]
        );
        if (!$admin->hasRole('Admin')) {
            $admin->assignRole('Admin');
        }

        // مثال: إنشاء Team Lead للتجربة
        if ($tlEmail = env('SEED_TL_EMAIL')) {
            $lead = User::firstOrCreate(
                ['email' => $tlEmail],
                ['name' => 'Team Lead', 'password' => Hash::make(env('SEED_TL_PASSWORD', 'password'))]
            );
            if (!$lead->hasRole('Team Lead')) {
                $lead->assignRole('Team Lead');
            }
        }

        // مثال: إنشاء HR Manager للتجربة
        if ($hrEmail = env('SEED_HR_EMAIL')) {
            $hr = User::firstOrCreate(
                ['email' => $hrEmail],
                ['name' => 'HR Manager', 'password' => Hash::make(env('SEED_HR_PASSWORD', 'password'))]
            );
            if (!$hr->hasRole('HR Manager')) {
                $hr->assignRole('HR Manager');
            }
        }

        // رجّع كاش الصلاحيات
        app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();
    }
}
