// ===============================
//  echo.js — Neon Toaster Version
// ===============================

import Echo from "laravel-echo";
import Pusher from "pusher-js";

window.Pusher = Pusher;
Pusher.logToConsole = false;

// === إعدادات الاتصال بـ Pusher ===
const CSRF =
    document
        .querySelector('meta[name="csrf-token"]')
        ?.getAttribute("content") || "";
const BASE = (window.APP_BASE_URL || "").replace(/\/+$/, "");
const AUTH_ENDPOINT = BASE ? `${BASE}/broadcasting/auth` : "/broadcasting/auth";

window.Echo = new Echo({
    broadcaster: "pusher",
    key: import.meta.env.VITE_PUSHER_APP_KEY,
    cluster: import.meta.env.VITE_PUSHER_APP_CLUSTER,
    forceTLS: true,
    authEndpoint: AUTH_ENDPOINT,
    auth: {
        headers: {
            "X-CSRF-TOKEN": CSRF,
            "X-Requested-With": "XMLHttpRequest",
            Accept: "application/json",
        },
    },
});

// ===============================
//  Helpers
// ===============================
function escapeHtml(s = "") {
    return s.replace(
        /[&<>"']/g,
        (m) =>
            ({
                "&": "&amp;",
                "<": "&lt;",
                ">": "&gt;",
                '"': "&quot;",
                "'": "&#039;",
            }[m])
    );
}

function timeAgo(iso) {
    try {
        const d = iso ? new Date(iso) : new Date();
        const s = (Date.now() - d.getTime()) / 1000;
        const step = [
            ["ث", 60],
            ["د", 60],
            ["س", 24],
            ["ي", 30],
            ["ش", 12],
            ["سنة", Infinity],
        ];
        let n = s,
            i = 0;
        for (; i < step.length - 1 && n >= step[i][1]; i++) n /= step[i][1];
        return `${Math.floor(n)}${step[i][0]} مضت`;
    } catch {
        return "الآن";
    }
}

function iconByPayload(p) {
    const model = (p.model || "").toLowerCase();
    if (model === "vacation") return "bi-umbrella";
    if (model === "attendance_permit") return "bi-alarm";
    return "bi-bell";
}

function toAbs(href) {
    if (!href) return "#";
    if (/^https?:\/\//i.test(href)) return href;
    const base = BASE || `${location.origin}/`;
    return `${base.replace(/\/+$/, "")}/${String(href).replace(/^\/+/, "")}`;
}

function buildHref(p) {
    if (p?.model && p?.model_id != null) {
        const m = String(p.model).toLowerCase();
        return `${(BASE || "").replace(
            /\/+$/,
            ""
        )}/notifications/open?m=${encodeURIComponent(m)}:${encodeURIComponent(
            p.model_id
        )}`;
    }
    return toAbs(p.open_url || "/");
}

function renderItem(p) {
    const a = document.createElement("a");
    a.href = buildHref(p);
    a.className =
        "d-flex align-items-center py-10px dropdown-item text-wrap fw-semibold";
    a.innerHTML = `
    <div class="fs-20px w-20px"><i class="bi ${iconByPayload(
        p
    )} text-theme"></i></div>
    <div class="flex-1 flex-wrap ps-3">
      <div class="mb-1 text-inverse">${escapeHtml(p.title || "تنبيه")}</div>
      <div class="small text-inverse text-opacity-50">${timeAgo(
          p.created_at
      )}</div>
    </div>
    <div class="ps-2 fs-16px"><i class="bi bi-chevron-right"></i></div>`;
    return a;
}

function existsInList(p) {
    try {
        const list = document.querySelector("#notif-list");
        if (!list) return false;
        const model = String(p.model || "").toLowerCase();
        const id = String(p.model_id ?? "");
        if (!model || !id) return false;
        const needle1 = `m=${encodeURIComponent(model)}%3A${encodeURIComponent(
            id
        )}`;
        const needle2 = `/dashboard/attendance/permits/${id}`;
        for (const a of list.querySelectorAll("a[href]")) {
            const href = a.getAttribute("href") || "";
            if (href.includes(needle2) || href.includes(needle1)) return true;
        }
        return false;
    } catch {
        return false;
    }
}

function incBadge() {
    const b = document.querySelector("#notif-badge");
    if (!b) return;
    const n = parseInt(b.textContent || "0", 10) + 1;
    b.textContent = String(n);
}

function setBadge(n) {
    const b = document.querySelector("#notif-badge");
    if (b) b.textContent = String(n || 0);
}

// ===============================
//  Toast Wrapper
// ===============================
function showToast(title, html, variant = "info") {
    if (window.toast) {
        window.toast[variant](title, html, 6000);
    } else {
        console.log(`[${variant.toUpperCase()}] ${title}: ${html}`);
    }
}

// ===============================
//  Main Logic
// ===============================
document.addEventListener("DOMContentLoaded", () => {
    if (!window.userId) return;

    const list = document.querySelector("#notif-list");
    const empty = document.querySelector("#notif-empty");

    // Diagnostic
    try {
        const p = window.Echo.connector.pusher;
        p.connection.bind("state_change", (s) =>
            console.log("Pusher state:", s)
        );
        p.bind("pusher:subscription_error", (e) =>
            console.error("Subscription error:", e)
        );
        p.bind("pusher:error", (e) => console.error("Pusher error:", e));
    } catch {}

    // === Vacation Events ===
    window.Echo.private(`vacations.${window.userId}`)
        .listen(".VacationRequested", (e) => {
            console.log("📢 VacationRequested:", e);
            showToast(
                "طلب إجازة جديد",
                `من: <b>${escapeHtml(e.employee)}</b><br>النوع: ${escapeHtml(
                    e.type || "—"
                )}<br>${escapeHtml(e.start)} → ${escapeHtml(e.end)}`,
                "info"
            );
        })
        .listen(".VacationStatusChanged", (e) => {
            console.log("📢 VacationStatusChanged:", e);
            showToast(
                "تحديث حالة إجازة",
                `#${e.vacation_id} — الحالة: <b>${escapeHtml(e.status)}</b>`,
                e.status === "rejected" ? "error" : "success"
            );
        });

    // === Permit Events ===
    window.Echo.private(`permits.${window.userId}`)
        .listen(".PermitRequested", (e) => {
            console.log("📢 PermitRequested:", e);
            const p = {
                model: "attendance_permit",
                model_id: e.permit_id,
                open_url: `/dashboard/attendance/permits/${e.permit_id}`,
                title: `طلب إذن جديد (#${e.permit_id})`,
                body: `من: ${e.employee || "موظف"} — النوع: ${e.type || "—"}`,
                created_at: new Date().toISOString(),
            };
            if (empty) empty.remove();
            if (!existsInList(p) && list) list.prepend(renderItem(p));
            incBadge();
            showToast(
                "طلب إذن جديد",
                `من: <b>${escapeHtml(
                    e.employee || "موظف"
                )}</b><br>النوع: ${escapeHtml(e.type || "—")}<br>${escapeHtml(
                    e.start || ""
                )} → ${escapeHtml(e.end || "")}`,
                "info"
            );
        })
        .listen(".PermitStatusChanged", (e) => {
            console.log("📢 PermitStatusChanged:", e);
            const p2 = {
                model: "attendance_permit",
                model_id: e.permit_id,
                open_url: `/dashboard/attendance/permits/${e.permit_id}`,
                title: `تحديث حالة إذن (#${e.permit_id})`,
                body: `الحالة: ${e.status || ""} — بواسطة ${
                    e.actor_name || ""
                }`,
                created_at: new Date().toISOString(),
            };
            if (empty) empty.remove();
            if (!existsInList(p2) && list) list.prepend(renderItem(p2));
            incBadge();
            showToast(
                "تحديث حالة إذن",
                `#${e.permit_id} — الحالة: <b>${escapeHtml(
                    e.status || ""
                )}</b><br>من: ${escapeHtml(e.actor_name || "")}`,
                e.status === "rejected" ? "error" : "success"
            );
        });

    // === Native Laravel Notifications ===
    window.Echo.private(`App.Models.User.${window.userId}`).notification(
        (payload) => {
            console.log("🔔 Notification:", payload);
            if (empty) empty.remove();
            if (!existsInList(payload) && list)
                list.prepend(renderItem(payload));
            incBadge();
            showToast(payload.title || "تنبيه", payload.body || "", "info");
        }
    );

    // Set Initial Badge Count
    setBadge(window.initialUnreadCount || 0);
});
