@extends('layout.default')
@section('title', 'Departments')


@include('partial.theme.neon-vars')
@include('partial.toast-js')

@push('css')
    <style>
        /* ===== Neon / HUD Tokens ===== */
        :root {
            --text-main: #dfe7ff;
            --muted: #8ea2c6;
            --neon-1: #2afc98;
            --neon-2: #00e0ff;
            --neon-3: #ff5fff;
            --border: rgba(0, 224, 255, .18);
            --border-2: rgba(0, 224, 255, .28);
            --glow: 0 0 12px rgba(42, 252, 152, .35), 0 0 30px rgba(0, 224, 255, .18);
            --ring: 0 0 0 2px rgba(0, 224, 255, .32), 0 0 30px rgba(0, 224, 255, .18) inset;
        }

        .hud-card {
            /* background: #28292b69; */
            border: 1px solid var(--border);
            border-radius: 14px;
            box-shadow: var(--glow);
        }

        .hud-head {
            border-bottom: 1px dashed var(--border);
            /* background: linear-gradient(180deg, rgba(0, 224, 255, .08), transparent); */
        }

        .neon-title {
            font-weight: 800;
            letter-spacing: .4px;
            margin: 0;
            background: linear-gradient(90deg, var(--neon-2), var(--neon-3));
            -webkit-background-clip: text;
            background-clip: text;
            color: transparent;
            text-shadow: 0 0 12px rgba(0, 224, 255, .28);
        }

        .hud-btn {
            background: linear-gradient(90deg, rgba(0, 224, 255, .14), rgba(255, 95, 255, .14));
            color: #e9f6ff;
            border: 1px solid var(--border-2);
            border-radius: 10px;
            padding: .45rem .7rem;
            box-shadow: var(--glow);
        }

        .hud-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 0 20px rgba(0, 224, 255, .35);
        }

        .hud-btn-icon {
            width: 34px;
            height: 34px;
            display: inline-grid;
            place-items: center;
            padding: 0;
            text-decoration: none
        }

        /* Table */
        .hud-table {
            color: #dfe7ff;
            margin: 0;
        }

        .hud-table thead th {
            border-top: 0;
            background: rgba(0, 224, 255, .06);
            color: #cfe7ff;
            text-transform: uppercase;
            font-size: .75rem;
            letter-spacing: .4px;
        }

        .hud-table tbody td {
            border-top: 1px solid var(--border);
            vertical-align: middle;
        }

        .badge-chip {
            background: rgba(0, 224, 255, .08);
            border: 1px solid var(--border-2);
            color: #cfe7ff;
            border-radius: 999px;
            padding: .25rem .55rem;
        }

        .badge-color {
            display: inline-block;
            border-radius: 8px;
            font-weight: 600;
            padding: .2rem .45rem;
            border: 1px solid rgba(255, 255, 255, .2);
            color: #10151f;
            text-shadow: 0 1px 0 rgba(255, 255, 255, .25);
        }

        .breadcrumb>li>a {
            color: #9fcfff
        }

        /* Modals */
        .modal-content {
            /* background: #141a24; */
            border: 1px solid var(--border);
            box-shadow: var(--glow);
            border-radius: 14px;
        }

        .form-control,
        .form-select {
            background: rgba(0, 224, 255, .07);
            border: 1px solid var(--border);
            color: #e3eeff;
            border-radius: 10px;
        }

        .form-control:focus,
        .form-select:focus {
            box-shadow: var(--ring);
            border-color: var(--border-2);
            color: #fff;
        }

        .form-label {
            color: var(--text-main);
            font-weight: 600;
        }


        @media (max-width: 991.98px) {
            .hud-head {
                padding: .75rem 1rem !important
            }
        }
    </style>
@endpush

@section('content')
    <div class="container-fluid">
        <meta name="csrf-token" content="{{ csrf_token() }}">

        @if (session('success'))
            <div class="alert alert-success alert-dismissible fade show">
                {{ session('success') }}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        @endif
        @if (session('error'))
            <div class="alert alert-danger alert-dismissible fade show">
                {{ session('error') }}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        @endif

        <!-- BEGIN breadcrumb -->

        <!-- END breadcrumb -->

        <!-- BEGIN page-header -->
        <h1 class="page-header neon-title">Departments Management</h1>
        <!-- END page-header -->

        <div class="row">
            <div class="col-12">
                <div class="hud-card">
                    <div class="hud-head d-flex justify-content-between align-items-center p-3">
                        <h4 class="m-0">Departments List</h4>
                        @if (auth()->user()->role === 'admin')
                            <button class="hud-btn hud-btn-icon" data-bs-toggle="modal" data-bs-target="#modalAddDepartment"
                                title="Add department">
                                <i class="fa fa-plus"></i>
                            </button>
                        @endif
                    </div>

                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table hud-table" id="departmentsTable">
                                <thead>
                                    <tr>
                                        <th>Name</th>
                                        <th>Code</th>
                                        <th>Color</th>
                                        <th>Parent Department</th>
                                        <th>Employees Count</th>
                                        <th class="text-end">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach ($departments as $dept)
                                        <tr id="dept-row-{{ $dept->id }}">
                                            <td>{{ $dept->name }}</td>
                                            <td><span class="badge-chip">{{ $dept->code }}</span></td>
                                            <td>
                                                <span class="badge-color"
                                                    style="background: {{ $dept->color_code }}">{{ $dept->color_code }}</span>
                                            </td>
                                            <td>{{ $dept->parent ? $dept->parent->name : '-' }}</td>
                                            <td><span class="badge-chip">{{ $dept->employees_count }} employees</span></td>
                                            <td class="text-end">
                                                <div class="btn-group" role="group">
                                                    <button class="hud-btn hud-btn-icon"
                                                        onclick="showEmployees({{ $dept->id }})"
                                                        title="View Employees">
                                                        <i class="fa fa-users"></i>
                                                    </button>
                                                    <button class="hud-btn hud-btn-icon"
                                                        onclick="showAttendance({{ $dept->id }})"
                                                        title="View Attendance">
                                                        <i class="fa fa-clock"></i>
                                                    </button>
                                                    <a href="{{ route('departments.show', $dept->id) }}"
                                                        class="hud-btn hud-btn-icon" title="View Details">
                                                        <i class="fa fa-eye"></i>
                                                    </a>
                                                    @if (auth()->user()->role === 'admin')
                                                        <button class="hud-btn hud-btn-icon"
                                                            onclick="editDepartment({{ $dept->id }}, '{{ $dept->name }}', '{{ $dept->code }}', '{{ $dept->color_code }}', {{ $dept->parent_id ?? 'null' }})"
                                                            title="Edit">
                                                            <i class="fa fa-edit"></i>
                                                        </button>
                                                        <button class="hud-btn hud-btn-icon"
                                                            onclick="deleteDepartment({{ $dept->id }})" title="Delete">
                                                            <i class="fa fa-trash"></i>
                                                        </button>
                                                    @endif
                                                </div>
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        @if (auth()->user()->role === 'admin')
            <!-- Add Department Modal -->
            <div class="modal fade" id="modalAddDepartment">
                <div class="modal-dialog">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h4 class="modal-title">Add Department</h4>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <form id="addDepartmentForm" action="{{ route('departments.store') }}" method="POST">
                                @csrf
                                <div class="mb-3">
                                    <label class="form-label">Name</label>
                                    <input type="text" class="form-control" name="name" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Code</label>
                                    <input type="text" class="form-control" name="code" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Color</label>
                                    <input type="color" class="form-control" name="color_code" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Parent Department</label>
                                    <select class="form-select" name="parent_id">
                                        <option value="">None</option>
                                        @foreach ($departments as $dept)
                                            <option value="{{ $dept->id }}">{{ $dept->name }}</option>
                                        @endforeach
                                    </select>
                                </div>
                            </form>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Close</button>
                            <button type="submit" class="hud-btn" form="addDepartmentForm">Add Department</button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Edit Department Modal -->
            <div class="modal fade" id="modalEditDepartment">
                <div class="modal-dialog">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h4 class="modal-title">Edit Department</h4>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <form id="editDepartmentForm" method="POST">
                                @csrf @method('PUT')
                                <div class="mb-3">
                                    <label class="form-label">Name</label>
                                    <input type="text" class="form-control" name="name" id="editDepartmentName"
                                        required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Code</label>
                                    <input type="text" class="form-control" name="code" id="editDepartmentCode"
                                        required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Color</label>
                                    <input type="color" class="form-control" name="color_code"
                                        id="editDepartmentColor" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Parent Department</label>
                                    <select class="form-select" name="parent_id" id="editDepartmentParent">
                                        <option value="">None</option>
                                        @foreach ($departments as $dept)
                                            <option value="{{ $dept->id }}">{{ $dept->name }}</option>
                                        @endforeach
                                    </select>
                                </div>
                            </form>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-outline-secondary"
                                data-bs-dismiss="modal">Close</button>
                            <button type="submit" class="hud-btn" form="editDepartmentForm">Update Department</button>
                        </div>
                    </div>
                </div>
            </div>
        @endif

        <!-- Employees Modal -->
        <div class="modal fade" id="modalEmployees" tabindex="-1">
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h4 class="modal-title" id="employeesModalLabel">Department Employees</h4>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body" id="employeesModalBody">
                        <div class="text-center py-4">
                            <div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Attendance Modal -->
        <div class="modal fade" id="modalAttendance" tabindex="-1">
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h4 class="modal-title" id="attendanceModalLabel">Department Attendance</h4>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="attendanceDate" class="form-label">Select Date</label>
                            <input type="date" class="form-control" id="attendanceDate" value="{{ date('Y-m-d') }}">
                        </div>
                        <div id="attendanceModalBody">
                            <div class="text-center py-4">
                                <div class="spinner-border" role="status"><span
                                        class="visually-hidden">Loading...</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('js')
    <script>
        function showEmployees(departmentId) {
            const modal = new bootstrap.Modal(document.getElementById('modalEmployees'));
            modal.show();

            document.getElementById('employeesModalBody').innerHTML =
                `<div class="text-center py-4"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div></div>`;

            fetch(`/departments/${departmentId}/employees`)
                .then(r => r.json())
                .then(data => {
                    let html = `<h6 class="mb-3">Department: ${data.department}</h6>`;
                    if (!data.employees.length) {
                        html += '<div class="alert alert-info">No employees found in this department.</div>';
                    } else {
                        html += `<div class="table-responsive">
          <table class="table hud-table">
            <thead><tr>
              <th>Employee Code</th><th>Name</th><th>Email</th><th>Status</th><th>Hire Date</th>
            </tr></thead><tbody>`;
                        data.employees.forEach(e => {
                            const ok = e.status === 'active';
                            html += `<tr>
            <td>${e.employee_code || 'N/A'}</td>
            <td>${e.name || 'N/A'}</td>
            <td>${e.email || 'N/A'}</td>
            <td><span class="badge ${ok ? 'bg-success' : 'bg-danger'}">${ok?'Active':'Inactive'}</span></td>
            <td>${e.hire_date || 'N/A'}</td>
          </tr>`;
                        });
                        html += `</tbody></table></div>`;
                    }
                    document.getElementById('employeesModalBody').innerHTML = html;
                })
                .catch(() => {
                    document.getElementById('employeesModalBody').innerHTML =
                        '<div class="alert alert-danger">Error loading department employees.</div>';
                });
        }

        function showAttendance(departmentId) {
            const modal = new bootstrap.Modal(document.getElementById('modalAttendance'));
            modal.show();
            document.getElementById('attendanceDate').setAttribute('data-department-id', departmentId);
            loadAttendance(departmentId, document.getElementById('attendanceDate').value);
        }

        function loadAttendance(departmentId, date) {
            document.getElementById('attendanceModalBody').innerHTML =
                `<div class="text-center py-4"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div></div>`;

            fetch(`/departments/${departmentId}/attendance?date=${encodeURIComponent(date)}`)
                .then(r => r.json())
                .then(data => {
                    let html = `<h6 class="mb-3">Department: ${data.department} — Date: ${data.date}</h6>`;
                    if (!data.attendance.length) {
                        html += '<div class="alert alert-info">No attendance records found for this date.</div>';
                    } else {
                        html += `<div class="table-responsive"><table class="table hud-table">
          <thead><tr>
            <th>Employee Code</th><th>Employee Name</th><th>Check In</th><th>Check Out</th><th>Total Hours</th>
          </tr></thead><tbody>`;
                        data.attendance.forEach(r => {
                            html += `<tr>
            <td>${r.employee.code}</td><td>${r.employee.name}</td>
            <td>${r.check_in || '-'}</td><td>${r.check_out || '-'}</td><td>${r.total_hours}</td>
          </tr>`;
                        });
                        html += `</tbody></table></div>`;
                    }
                    document.getElementById('attendanceModalBody').innerHTML = html;
                })
                .catch(() => {
                    document.getElementById('attendanceModalBody').innerHTML =
                        '<div class="alert alert-danger">Error loading attendance data.</div>';
                });
        }

        @if (auth()->user()->role === 'admin')
            function editDepartment(id, name, code, color, parentId) {
                const form = document.getElementById('editDepartmentForm');
                form.action = `/departments/${id}`;
                document.getElementById('editDepartmentName').value = name;
                document.getElementById('editDepartmentCode').value = code;
                document.getElementById('editDepartmentColor').value = color;
                document.getElementById('editDepartmentParent').value = parentId || '';
                new bootstrap.Modal(document.getElementById('modalEditDepartment')).show();
            }

            function deleteDepartment(id) {
                if (!confirm('Are you sure you want to delete this department?')) return;
                fetch(`/departments/${id}`, {
                        method: 'DELETE',
                        headers: {
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                            'Content-Type': 'application/json',
                        },
                    })
                    .then(r => r.json())
                    .then(data => data.success ? location.reload() : alert(data.error || 'Error deleting department'))
                    .catch(() => alert('Error deleting department'));
            }
        @endif

        document.addEventListener('DOMContentLoaded', () => {
            document.getElementById('attendanceDate').addEventListener('change', function() {
                const deptId = this.getAttribute('data-department-id');
                if (deptId) loadAttendance(deptId, this.value);
            });
        });
    </script>
@endpush
