@extends('layout.default')
@section('title', 'Role Management')

@section('content')
<div class="container-fluid">
  <div class="d-flex align-items-center flex-wrap mb-2">
    <h4 class="mb-0 me-auto">Role Management</h4>
  </div>

  @include('partial.datatable.assets')        {{-- أصول DT + ثيم + هيلبر، مرة واحدة --}}
  @include('pages.roles.partial.filters')           {{-- الفلاتر --}}

  <div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
      <strong>Users</strong>
      <div class="bulk-actions d-flex gap-2 align-items-center flex-sm-row flex-column">
        <select id="bulkRole" class="form-select form-select-sm flex-fill" style="max-width:220px">
          @foreach($roles as $r)
            <option value="{{ $r->name }}">{{ $r->name }}</option>
          @endforeach
        </select>
        <div class="d-grid gap-2 d-sm-flex w-100 w-sm-auto">
          <button id="bulkAttach" class="btn btn-sm btn-neon btn-neon-primary w-100 w-sm-auto">Bulk Attach</button>
          <button id="bulkDetach" class="btn btn-sm btn-neon btn-neon-outline w-100 w-sm-auto">Bulk Detach</button>
        </div>
      </div>
    </div>

    <div class="card-body">
      @include('partial.datatable.init', [
        'id' => 'usersTable',
        'ajax' => route('roles.dt'),              // Endpoint JSON
        'columns' => [
          ['data'=>null, 'orderable'=>false, 'searchable'=>false, 'render'=>'checkbox'],
          ['data'=>'id'],
          ['data'=>'name'],
          ['data'=>'email'],
          ['data'=>null, 'orderable'=>false, 'searchable'=>false, 'render'=>'roleBadges'],
          ['data'=>null, 'orderable'=>false, 'searchable'=>false, 'render'=>'roleActions'],
        ],
        'order' => [[2,'asc']],
        'exportColumns' => [1,2,3,4],
        'responsive' => 'inline',
        'filtersHook' => 'rolesFiltersHook',
        'thead' => view('pages.roles.partial.thead')
      ])
    </div>
  </div>
</div>

{{-- Toast container لو مش في الـlayout --}}
<div class="toast-container position-fixed top-0 end-0 p-3" id="toastContainer" style="z-index:1080;"></div>
@endsection

@push('js')
<script>
  // ===== 1) فلاتر الأجاكس
  window.rolesFiltersHook = function(){
    return {
      roles: Array.from(document.getElementById('filterRoles')?.selectedOptions||[]).map(o=>o.value),
      has: (document.getElementById('filterHas')||{}).value || 'all',
      departments: Array.from(document.getElementById('filterDepts')?.selectedOptions||[]).map(o=>o.value),
      search: document.getElementById('globalSearch')?.value || ''
    };
  };

  // ===== 2) Renderers
  window.DT.renderers.roleBadges = function(_, row){
    const roles = row.roles || [];
    if(!roles.length) return '<span class="text-muted">—</span>';
    return roles.map(r=>
      `<span class="badge-role" data-role="${r}">${r} <button class="btn-x js-revoke" data-role="${r}">×</button></span>`
    ).join(' ');
  };

  window.DT.renderers.roleActions = function(_, row){
    const roleOptions = @json($roles->pluck('name')).map(r=>`<option value="${r}">${r}</option>`).join('');
    return `
      <div class="d-flex justify-content-end gap-2">
        <select class="form-select form-select-sm role-select select-sm" data-id="${row.id}">
          ${roleOptions}
        </select>
        <button class="btn btn-sm btn-neon btn-neon-primary js-attach" data-id="${row.id}">Attach</button>
      </div>`;
  };

  // ===== 3) Handlers (attach/revoke/bulk/search/reset/selectAll)
  (function(){
    const CSRF  = document.querySelector('meta[name="csrf-token"]')?.content;
    const ROUTES = {
      attach:    @json(route('roles.attach')),
      detach:    @json(route('roles.detach')),
      bulkAttach:@json(route('roles.bulk.attach')),
      bulkDetach:@json(route('roles.bulk.detach')),
    };
    const table = ()=> window.DT_INSTANCES['usersTable'];

    // Search + filters
    const $search = document.getElementById('globalSearch');
    const reload = ()=> table()?.ajax?.reload();
    let st;
    $search && $search.addEventListener('input', ()=>{ clearTimeout(st); st=setTimeout(reload, 300); });
    ['filterHas','filterRoles','filterDepts'].forEach(id=>{
      document.getElementById(id)?.addEventListener('change', reload);
    });
    document.getElementById('btnReset')?.addEventListener('click', ()=>{
      if($search) $search.value='';
      const fr = document.getElementById('filterRoles'); fr && Array.from(fr.options).forEach(o=>o.selected=false);
      const fd = document.getElementById('filterDepts'); fd && Array.from(fd.options).forEach(o=>o.selected=false);
      const fh = document.getElementById('filterHas');   if(fh) fh.value='all';
      reload();
    });

    // Select all
    const $checkAll = document.getElementById('checkAll');
    $checkAll?.addEventListener('change', (e)=>{
      document.querySelectorAll('#usersTable .row-check').forEach(c=> c.checked=e.target.checked);
    });

    // Delegate actions (attach/revoke)
    document.getElementById('usersTable').addEventListener('click', async (e)=>{
      const tr = e.target.closest('tr'); if(!tr) return;
      const row = table()?.row(tr).data(); if(!row) return;

      // Attach
      if (e.target.classList.contains('js-attach')) {
        const sel  = tr.querySelector('.role-select');
        const role = sel?.value;
        try{
          const fd = new FormData(); fd.append('user_id', row.id); fd.append('role', role);
          const res = await fetch(ROUTES.attach, {method:'POST', headers:{'X-CSRF-TOKEN':CSRF,'Accept':'application/json','X-Requested-With':'XMLHttpRequest'}, body:fd});
          const j = await res.json(); if(!res.ok || j.success===false) throw new Error(j.message||'Attach failed');
          table().ajax.reload(null,false); window.toast?.success?.('Attached', `${role} → #${row.id}`);
        }catch(err){ window.toast?.error?.('Attach Failed', err.message||'Error'); }
      }

      // Revoke
      if (e.target.classList.contains('js-revoke')) {
        const role = e.target.getAttribute('data-role');
        if (!confirm(`Revoke role "${role}" from user #${row.id}?`)) return;
        try{
          const fd = new FormData(); fd.append('user_id', row.id); fd.append('role', role);
          const res = await fetch(ROUTES.detach, {method:'POST', headers:{'X-CSRF-TOKEN':CSRF,'Accept':'application/json','X-Requested-With':'XMLHttpRequest'}, body:fd});
          const j = await res.json(); if(!res.ok || j.success===false) throw new Error(j.message||'Revoke failed');
          table().ajax.reload(null,false); window.toast?.warning?.('Revoked', `${role} ← #${row.id}`);
        }catch(err){ window.toast?.error?.('Revoke Failed', err.message||'Error'); }
      }
    });

    // Bulk
    async function runBulk(url){
      const ids  = Array.from(document.querySelectorAll('#usersTable .row-check:checked')).map(c=>c.value);
      const role = document.getElementById('bulkRole')?.value;
      if (!ids.length) return window.toast?.info?.('No selection','Select at least one user.');
      if (!role)       return window.toast?.info?.('Pick role','Choose a role first.');
      if (!confirm(`Run bulk on ${ids.length} users?`)) return;
      const fd = new FormData(); ids.forEach(id=>fd.append('user_ids[]', id)); fd.append('role', role);
      try{
        const res = await fetch(url, {method:'POST', headers:{'X-CSRF-TOKEN':CSRF,'Accept':'application/json','X-Requested-With':'XMLHttpRequest'}, body:fd});
        const j = await res.json(); if(!res.ok || j.success===false) throw new Error(j.message||'Bulk failed');
        table().ajax.reload(null,false); window.toast?.success?.('Done', j.message||'Bulk action executed.');
      }catch(err){ window.toast?.error?.('Bulk Failed', err.message||'Error'); }
    }
    document.getElementById('bulkAttach')?.addEventListener('click', ()=> runBulk(@json(route('roles.bulk.attach'))));
    document.getElementById('bulkDetach')?.addEventListener('click', ()=> runBulk(@json(route('roles.bulk.detach'))));
  })();
</script>
@endpush
