<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Department;
use App\Models\TeamMember;
use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;

class DepartmentsTeamsPageTest extends TestCase
{
    use RefreshDatabase;

    protected $admin;
    protected $user;

    protected function setUp(): void
    {
        parent::setUp();

        $this->admin = User::factory()->create(['role' => 'admin']);
        $this->user = User::factory()->create(['role' => 'user']);
    }

    /** @test */
    public function page_requires_authentication()
    {
        $response = $this->get('/departments-teams');
        $response->assertRedirect('/login');
    }

    /** @test */
    public function authenticated_users_can_view_page()
    {
        $this->actingAs($this->user);

        $response = $this->get('/departments-teams');
        $response->assertStatus(200)
            ->assertViewIs('pages.departments-teams.index');
    }

    /** @test */
    public function page_shows_departments_table()
    {
        $this->actingAs($this->user);

        $department = Department::factory()->create([
            'name' => 'Test Department',
            'color_code' => '#FF0000'
        ]);

        $response = $this->get('/departments-teams');
        $response->assertStatus(200)
            ->assertSee('Test Department');
    }

    /** @test */
    public function page_shows_team_members_table()
    {
        $this->actingAs($this->user);

        $department = Department::factory()->create();
        $member = TeamMember::factory()->create([
            'name' => 'John Doe',
            'department_id' => $department->id
        ]);

        $response = $this->get('/departments-teams');
        $response->assertStatus(200)
            ->assertSee('John Doe');
    }

    /** @test */
    public function only_admin_sees_add_buttons()
    {
        // Test as regular user
        $this->actingAs($this->user);
        $response = $this->get('/departments-teams');
        $response->assertStatus(200)
            ->assertDontSee('data-bs-target="#modalAddDepartment"')
            ->assertDontSee('data-bs-target="#modalAddTeamMember"');

        // Test as admin
        $this->actingAs($this->admin);
        $response = $this->get('/departments-teams');
        $response->assertStatus(200)
            ->assertSee('data-bs-target="#modalAddDepartment"')
            ->assertSee('data-bs-target="#modalAddTeamMember"');
    }

    /** @test */
    public function shows_department_hierarchy()
    {
        $this->actingAs($this->user);

        $parent = Department::factory()->create(['name' => 'Parent Dept']);
        $child = Department::factory()->create([
            'name' => 'Child Dept',
            'parent_id' => $parent->id
        ]);

        $response = $this->get('/departments-teams');
        $response->assertStatus(200)
            ->assertSee('Parent Dept')
            ->assertSee('Child Dept');
    }

    /** @test */
    public function shows_color_coded_departments()
    {
        $this->actingAs($this->user);

        $department = Department::factory()->create([
            'name' => 'Colored Dept',
            'color_code' => '#FF0000'
        ]);

        $response = $this->get('/departments-teams');
        $response->assertStatus(200)
            ->assertSee('#FF0000')
            ->assertSee('Colored Dept');
    }
}
